from model_utils import FieldTracker
from model_utils.models import TimeStampedModel
from ordered_model.models import OrderedModel
from simple_history.models import HistoricalRecords

from django.contrib.auth import get_user_model
from django.db import models
from django.utils.translation import gettext_lazy as _

User = get_user_model()


class Skill(TimeStampedModel):
    name = models.CharField(_("название"), max_length=255)
    canonical_name = models.CharField(
        _("каноническое имя"), max_length=255, unique=True
    )

    class Meta:
        verbose_name = _("навык")
        verbose_name_plural = _("навыки")

    def __str__(self):
        return self.name


class Position(TimeStampedModel):
    name = models.CharField(_("название"), max_length=255, unique=True)

    class Meta:
        ordering = ("name",)
        verbose_name = _("Должность ментора")
        verbose_name_plural = _("Должности менторов")

    def __str__(self):
        return self.name


class MentorQuerySet(models.QuerySet):
    def active(self):
        return self.filter(is_published=True, user__is_active=True)

    def on_staff(self):
        return self.filter(user__staff_profile__is_dismissed=False)

    def visible(self):
        return self.active().on_staff()


class Mentor(TimeStampedModel):
    user = models.OneToOneField(
        User,
        on_delete=models.PROTECT,
        verbose_name=_("пользователь"),
    )
    description = models.TextField(_("описание"))
    assistance = models.JSONField(_("чем могу помочь"), blank=True, default=list)
    skills = models.ManyToManyField(
        Skill, verbose_name=_("навыки"), related_name="mentors", blank=True
    )
    position = models.ForeignKey(
        Position,
        verbose_name=_("должность"),
        related_name="mentors",
        blank=True,
        null=True,
        on_delete=models.PROTECT,
    )
    carrier_begin = models.DateField(
        _("Дата начала работы по профессии"), null=True, blank=True
    )
    feedback_count = models.IntegerField(_("количество отзывов"), default=0)
    mentees_count = models.IntegerField(_("количество менти"), default=0)

    is_published = models.BooleanField(_("виден в каталоге"), default=True)
    is_ready_for_mentorships = models.BooleanField(
        _("готов принимать запросы на менторство"), default=True
    )

    objects = MentorQuerySet.as_manager()

    @property
    def staff_profile(self):
        return self.user.staff_profile

    @property
    def is_visible(self) -> bool:
        """
        Показывать ментора в каталоге
        """
        return (
            self.is_published
            and self.user.is_active
            and not self.staff_profile.is_dismissed
        )

    class Meta:
        verbose_name = _("ментор")
        verbose_name_plural = _("менторы")

    def __str__(self):
        return str(self.pk)


class Mentorship(TimeStampedModel):
    class Status(models.TextChoices):
        CREATED = "CREATED", _("Создано")
        ACCEPTED = "ACCEPTED", _("Одобрено")
        DECLINED = "DECLINED", _("Отклонено")
        PAUSED = "PAUSED", _("Приостановлено")
        COMPLETED = "COMPLETED", _("Завершено")

    mentor = models.ForeignKey(
        Mentor,
        on_delete=models.PROTECT,
        verbose_name=_("ментор"),
        related_name="mentorships",
    )
    mentee = models.ForeignKey(
        User,
        on_delete=models.PROTECT,
        verbose_name=_("менти"),
        related_name="mentorships",
    )
    intro = models.TextField(_("запрос от менти"))
    status = models.CharField(
        max_length=255,
        verbose_name=_("статус"),
        choices=Status.choices,
        default=Status.CREATED,
    )
    status_by = models.ForeignKey(
        User,
        on_delete=models.PROTECT,
        verbose_name=_("статус выставил"),
        related_name="status_in_mentorships",
    )
    status_message = models.TextField(_("сообщение статуса"), blank=True)
    accepted_date = models.DateTimeField(_("начало менторства"), null=True, blank=True)
    completed_date = models.DateTimeField(_("конец менторства"), null=True, blank=True)
    removed_by_mentor = models.BooleanField(_("удалён у ментора"), default=False)
    removed_by_mentee = models.BooleanField(_("удалён у менти"), default=False)

    history = HistoricalRecords(_("история"))

    tracker = FieldTracker(fields=["status"])

    class Meta:
        verbose_name = _("менторство")
        verbose_name_plural = _("менторства")

    def __str__(self):
        return str(self.pk)

    def remove_for_mentor(self):
        self.removed_by_mentor = True

        self.save()

    def remove_for_mentee(self):
        self.removed_by_mentee = True

        self.save()


class FeedbackBaseQuerySet(models.QuerySet):
    def visible(self):
        return self.filter(is_visible=True, can_publish=True, is_published=True)


class FeedbackBase(TimeStampedModel):
    mentorship = models.ForeignKey(
        Mentorship,
        on_delete=models.PROTECT,
        verbose_name=_("менторство"),
        related_name="%(class)s",
    )
    comments = models.TextField(_("комментарии"))
    is_visible = models.BooleanField(_("виден пользователю"), default=False)
    can_publish = models.BooleanField(_("можно публиковать"), default=False)
    is_published = models.BooleanField(_("опубликован"), default=False)

    objects = FeedbackBaseQuerySet.as_manager()

    def publish(self):
        self.is_published = True

        self.save()

    def unpublish(self):
        self.is_published = False

        self.save()

    class Meta:
        abstract = True

    def __str__(self):
        return str(self.pk)


class MentorFeedback(FeedbackBase):
    mentor = models.ForeignKey(
        Mentor,
        on_delete=models.PROTECT,
        verbose_name=_("ментор"),
        related_name="feedback",
    )

    class Meta(FeedbackBase.Meta):
        verbose_name = _("отзыв на ментора")
        verbose_name_plural = _("отзывы на ментора")


class MenteeFeedback(FeedbackBase):
    mentee = models.ForeignKey(
        User,
        on_delete=models.PROTECT,
        verbose_name=_("менти"),
        related_name="feedback",
    )

    class Meta(FeedbackBase.Meta):
        verbose_name = _("отзыв на менти")
        verbose_name_plural = _("отзывы на менти")


class ExperienceType(OrderedModel, TimeStampedModel):
    name = models.CharField(_("название"), max_length=255)
    slug = models.SlugField(_("код"))

    class Meta:
        verbose_name = _("тип опыта")
        verbose_name_plural = _("типы опыта")

    def __str__(self):
        return self.name


class MentorExperience(TimeStampedModel):
    mentor = models.ForeignKey(
        Mentor,
        verbose_name=_("ментор"),
        on_delete=models.PROTECT,
        related_name="experiences",
    )
    experience_type = models.ForeignKey(
        ExperienceType,
        verbose_name=_("тип опыта"),
        on_delete=models.PROTECT,
        related_name="experiences",
    )
    begin = models.DateField(_("начало"), null=True, blank=True)
    end = models.DateField(_("конец"), null=True, blank=True)
    name = models.CharField(_("название"), max_length=255)
    description = models.TextField(_("описание"), blank=True)

    class Meta:
        verbose_name = verbose_name_plural = _("опыт ментора")

    def __str__(self):
        return str(self.pk)
