from typing import Optional

from django.db import models, transaction
from django.dispatch import receiver

from mentor.staff.models import StaffProfile
from mentor.users.enums import NotificationEvent
from mentor.utils.defer import defer_or_sync

from .models import MenteeFeedback, Mentor, MentorFeedback, Mentorship
from .services import (
    send_mentee_feedback_notification,
    send_mentor_feedback_notification,
    send_mentorship_notification,
    set_accepted_date,
    set_created_date,
    update_mentor_feedback_count,
    update_mentor_mentees_count,
    update_mentor_position,
)


def get_notification_event(
    instance: Mentorship, created: bool, **kwargs
) -> Optional[NotificationEvent]:
    """
    Возвращает событие для уведомления (если оно определено)
    """
    event = None

    # менти создал менторство
    if created and instance.mentee_id == instance.status_by_id:
        event = NotificationEvent.MENTEE_CREATED_MENTORSHIP

    elif instance.tracker.has_changed("status"):
        if instance.status_by_id == instance.mentee_id:

            # менти завершил менторство
            if instance.status == Mentorship.Status.COMPLETED:
                event = NotificationEvent.MENTEE_COMPLETED_MENTORSHIP

        elif instance.status_by_id == instance.mentor.user_id:

            # ментор подтвердил заявку на менторство
            if instance.status == Mentorship.Status.ACCEPTED:
                event = NotificationEvent.MENTOR_ACCEPTED_MENTORSHIP

            # ментор отклонил заявку на менторство
            elif instance.status == Mentorship.Status.DECLINED:
                event = NotificationEvent.MENTOR_DECLINED_MENTORSHIP

            # ментор завершил менторство
            elif instance.status == Mentorship.Status.COMPLETED:
                event = NotificationEvent.MENTOR_COMPLETED_MENTORSHIP

    return event


@receiver(signal=models.signals.post_save, sender=Mentorship)
def mentorship_post_save_handler(instance: Mentorship, created: bool, **kwargs):
    transaction.on_commit(
        lambda: defer_or_sync(
            update_mentor_mentees_count,
            {"kwargs": {"pk": instance.mentor_id}},
        )
    )

    event = get_notification_event(instance, created, **kwargs)
    if event:
        transaction.on_commit(
            lambda: defer_or_sync(
                send_mentorship_notification,
                {"kwargs": {"pk": instance.pk, "event": event}},
            )
        )


@receiver(signal=models.signals.post_delete, sender=Mentorship)
def mentorship_post_delete_handler(instance: Mentorship, **kwargs):
    transaction.on_commit(
        lambda: defer_or_sync(
            update_mentor_mentees_count,
            {"kwargs": {"pk": instance.mentor_id}},
        )
    )


@receiver(signal=models.signals.post_save, sender=MentorFeedback)
def mentor_feedback_post_save_handler(instance: MentorFeedback, created, **kwargs):
    transaction.on_commit(
        lambda: defer_or_sync(
            update_mentor_feedback_count,
            {"kwargs": {"pk": instance.mentor_id}},
        )
    )

    if created and instance.is_visible:
        transaction.on_commit(
            lambda: defer_or_sync(
                send_mentor_feedback_notification, {"kwargs": {"pk": instance.pk}}
            )
        )


@receiver(signal=models.signals.post_save, sender=MenteeFeedback)
def mentee_feedback_post_save_handler(instance: MenteeFeedback, created, **kwargs):
    if created and instance.is_visible:
        transaction.on_commit(
            lambda: defer_or_sync(
                send_mentee_feedback_notification, {"kwargs": {"pk": instance.pk}}
            )
        )


@receiver(signal=models.signals.post_delete, sender=MentorFeedback)
def feedback_post_delete_handler(instance: MentorFeedback, **kwargs):
    transaction.on_commit(
        lambda: defer_or_sync(
            update_mentor_feedback_count,
            {"kwargs": {"pk": instance.mentor_id}},
        )
    )


def check_staff_position_changed(instance: StaffProfile):
    original_position_ru = getattr(instance, "_original_position_ru", "")
    position_changed = original_position_ru != instance.position_ru
    if not position_changed:
        return

    mentor = getattr(instance.user, "mentor", None)
    if mentor:
        update_mentor_position(mentor, instance.position_ru)


@receiver(signal=models.signals.pre_save, sender=Mentorship)
def mentorship_pre_save_handler(instance: Mentorship, **kwargs):
    set_accepted_date(instance)
    set_created_date(instance)


@receiver(signal=models.signals.post_init, sender=StaffProfile)
def staffprofile_post_init_handler(instance: StaffProfile, **kwargs):
    instance._original_position_ru = instance.position_ru


@receiver(signal=models.signals.post_save, sender=StaffProfile)
def staffprofile_post_save_handler(instance: StaffProfile, **kwargs):
    check_staff_position_changed(instance)


@receiver(signal=models.signals.pre_save, sender=Mentor)
def mentor_pre_save_handler(instance: Mentor, **kwargs):
    update_mentor_position(
        mentor=instance,
        position_name=instance.staff_profile.position_ru,
        commit=False,
    )
