from django_object_actions import DjangoObjectActions

from django.contrib import admin, messages
from django.utils.html import format_html
from django.utils.translation import gettext_lazy as _

from .models import (
    GroupNode,
    StaffCity,
    StaffCountry,
    StaffGroup,
    StaffLeadership,
    StaffOffice,
    StaffProfile,
)


@admin.register(StaffGroup)
class StaffGroupAdmin(DjangoObjectActions, admin.ModelAdmin):
    list_display = ("name", "level", "parent", "is_active", "created", "modified")
    list_select_related = ("parent",)
    raw_id_fields = ("parent",)
    search_fields = ("name", "parent__name", "node__path")
    fields = (
        "parent",
        "name",
        "level",
        "url",
        "group_type",
        "is_active",
    )

    change_actions = [
        "update_group",
    ]

    def update_group(self, request, obj: StaffGroup):
        from .services import load_staff_groups

        try:
            load_staff_groups(group_ids=obj.pk)
            return self.message_user(
                request, _("Данные по подразделению были обновлены")
            )
        except Exception:
            return self.message_user(
                request, _("Ошибка при обновлении данных со Стаффа"), messages.ERROR
            )

    update_group.label = _("обновить со стаффа")


@admin.register(GroupNode)
class GroupNodeAdmin(admin.ModelAdmin):
    list_display = (
        "display_name_indent",
        "path",
    )
    list_select_related = ("group",)
    search_fields = ("group__name",)

    def display_name_indent(self, obj: GroupNode):
        return format_html(
            '<span style="padding-left: {0}px">{1}</span>',
            10 * (obj.depth - 1),
            str(obj.group),
        )

    display_name_indent.short_description = _("Подразделение")

    def has_change_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False


@admin.register(StaffOffice)
class StaffOfficeAdmin(admin.ModelAdmin):
    list_display = ("name", "is_active", "created", "modified")


@admin.register(StaffCity)
class StaffCityAdmin(admin.ModelAdmin):
    list_display = ("name", "is_active", "created", "modified")


@admin.register(StaffCountry)
class StaffCountryAdmin(admin.ModelAdmin):
    list_display = ("name", "is_active", "created", "modified")


class StaffLeadershipInline(admin.TabularInline):
    model = StaffLeadership
    fields = ("group", "role")
    raw_id_fields = ("group",)
    extra = 0

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.select_related("group")


class StaffGroupInline(admin.TabularInline):
    model = StaffProfile.groups.through
    fields = ("staffgroup", "display_group_level", "display_group_type")
    readonly_fields = ("staffgroup", "display_group_level", "display_group_type")
    verbose_name_plural = _("подразделения")
    verbose_name = _("подразделение")
    extra = 0

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.select_related("staffgroup")

    def display_group_level(self, obj: StaffProfile.groups):
        return obj.staffgroup.level

    display_group_level.short_description = _("уровень")

    def display_group_type(self, obj: StaffProfile.groups):
        return obj.staffgroup.get_group_type_display()

    display_group_type.short_description = _("тип")


@admin.register(StaffProfile)
class StaffProfileAdmin(DjangoObjectActions, admin.ModelAdmin):
    list_display = ("user", "joined_at", "created", "modified")
    list_select_related = ("user", "city", "office", "groups")
    fields = (
        "user",
        ("first_name_ru", "last_name_ru"),
        ("first_name_en", "last_name_en"),
        "position_ru",
        "position_en",
        "office",
        "city",
        "language_native",
        "joined_at",
        "is_active",
        "is_dismissed",
    )
    search_fields = (
        "user__username",
        "first_name_ru",
        "last_name_ru",
        "first_name_en",
        "last_name_en",
    )

    inlines = [
        StaffGroupInline,
        StaffLeadershipInline,
    ]

    change_actions = ["update_profile"]

    def get_queryset(self, request):
        qs = super().get_queryset(request)
        return qs.select_related("user", "city", "office")

    def update_profile(self, request, obj: StaffProfile):
        from .services import load_staff_profile

        if load_staff_profile(obj.user_id):
            return self.message_user(request, _("Данные пользователя были обновлены"))
        else:
            return self.message_user(
                request, _("Ошибка при обновлении данных со Стаффа"), messages.ERROR
            )

    update_profile.label = _("обновить со стаффа")

    def has_change_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False
