import logging
from collections import Counter
from datetime import datetime
from typing import Optional, Union

from django.contrib.auth import get_user_model
from django.db import transaction
from django.utils import timezone

from mentor.contrib.staff.client import staff_api

from .models import (
    StaffCity,
    StaffCountry,
    StaffGroup,
    StaffLeadership,
    StaffOffice,
    StaffProfile,
)
from .utils import lookup_field, parse_fields

User = get_user_model()

log = logging.getLogger(__name__)


STAFF_API_DEFAULT_LIMIT = 1000


class CacheManager:
    """
    Менеджер кэшей в памяти
    """

    def __init__(self):
        self._caches = {}

    def get(self, key, cast=dict):
        if not self._caches.get(key):
            self._caches[key] = cast()
        return self._caches[key]

    def clear(self, key) -> None:
        cache = self._caches.get(key)
        if cache:
            cache.clear()

    def clear_all(self) -> None:
        self._caches.clear()


class Counts(Counter):
    """
    Счетчики с информативным выводом
    """

    def __str__(self):
        return ", ".join([f"{k}: {v}" for k, v in self.items()])


class BaseLoader:
    """
    Базовый загрузчик со Стаффа
    """

    endpoint = None
    init_kwargs = None
    fields = None
    limit = STAFF_API_DEFAULT_LIMIT
    max_limit = None

    def __init__(self, limit=None, max_limit=None, **kwargs):
        if limit:
            self.limit = limit

        self.max_limit = max_limit
        self.cache = CacheManager()
        self.counts = Counts()

    def clear_cache(self):
        self.cache.clear_all()
        self.counts.clear()

    def get_init_kwargs(self) -> dict:
        return dict(self.init_kwargs or {})

    def get_fields(self):
        if self.fields:
            return parse_fields(self.fields)

    def request(self, **kwargs):
        params = self.get_init_kwargs() or {}
        params["_limit"] = self.limit

        fields = self.get_fields()
        if fields:
            params["_fields"] = ",".join(set(fields))

        if kwargs:
            params.update(**kwargs)

        response = getattr(staff_api, self.endpoint).get(**params)
        log.debug("staff api response: %r", response)

        return response

    def get_items(self, **kwargs):
        response = self.request(**kwargs)
        return self.process_data(response)

    def process_data(self, response, **kwargs):
        has_results = True

        while has_results:
            result = response.get("result", [])
            page = response.get("page", 1)
            pages = response.get("pages", 1)

            for item in result:
                yield item

            if self.max_limit:
                current_limit = self.limit * int(page)
                if current_limit >= self.max_limit:
                    break

            if page < pages:
                page += 1
                response = self.request(_page=page)
                continue

            has_results = False

        return False

    def update(self, **kwargs):
        return NotImplementedError("method update must be implemented")


class BaseProfileLoader(BaseLoader):
    """
    Базовый загрузчик профилей пользователей
    """

    uid_field = "uid"

    def __init__(self, uids: Union[list, int], **kwargs):
        if isinstance(uids, int):
            uids = [uids]
        self.uids = uids

        super().__init__(**kwargs)

    def get_init_kwargs(self) -> dict:
        kwargs = super().get_init_kwargs()
        kwargs[self.uid_field] = ",".join(map(str, self.uids))
        return kwargs

    def get_profile(self, uid: int, force_update: bool = False) -> "StaffProfile":
        cached_profiles = self.cache.get("profiles")
        if uid not in cached_profiles or force_update:
            profile = StaffProfile.objects.get(**{StaffProfile.YAUID_FIELD: uid})
            cached_profiles[uid] = profile
        else:
            self.counts["profile_hits"] += 1

        return cached_profiles[uid]

    def profile_data(self, **kwargs):
        for item in self.get_items():
            uid = lookup_field(item, self.uid_field)
            if not uid:
                log.warning("No uid found: %r", item)
                continue

            yield self.get_profile(uid), item

    def update(self, **kwargs):
        for profile, data in self.profile_data():
            with transaction.atomic():
                self.update_profile(profile, data)
            self.counts["processed"] += 1

        log.info("Summary: %s", self.counts)
        self.clear_cache()

    def update_profile(self, profile: "StaffProfile", data: dict):
        return NotImplementedError("method update_profile must be implemented")


class ModelMixin:
    def method_or_create(self, model, method, data: dict, **kwargs):
        model_name = model.__name__.lower()
        pk = data["id"]
        created = False

        cache = self.cache.get(model_name, cast=set)
        no_cache = kwargs.pop("no_cache", False)

        if pk not in cache or no_cache:
            extracted = model.extract_from_dict(data)
            if kwargs:
                extracted.update(**kwargs)

            _, created = method(id=pk, defaults=extracted)
            cache.add(pk)
            self.counts[f"{model_name}_size"] += 1
        else:
            if not no_cache:
                self.counts[f"{model_name}_hits"] += 1

        if created:
            self.counts[f"{model_name}_created"] += 1

        return pk

    def get_or_create_model(self, model, data: dict, **kwargs):
        return self.method_or_create(
            model,
            model.objects.get_or_create,
            data,
            **kwargs,
        )

    def update_or_create_model(self, model, data: dict, **kwargs):
        return self.method_or_create(
            model,
            model.objects.update_or_create,
            data,
            **kwargs,
        )


class GroupMixin:
    """
    Миксин для обновления подразделений со Стаффа
    """

    @transaction.atomic()
    def update_or_create_group(self, group: dict, **kwargs):
        return self.update_or_create_model(
            StaffGroup,
            group,
            parent_id=self.get_parent_group(group, self.update_or_create_model),
            **kwargs,
        )

    @transaction.atomic()
    def get_or_create_group(self, group: dict, **kwargs):
        return self.get_or_create_model(
            StaffGroup,
            group,
            parent_id=self.get_parent_group(group, self.get_or_create_model),
            **kwargs,
        )

    def get_parent_group(self, group: dict, method_or_create: callable):
        parent_group_id = None

        for ancestor in group.get("ancestors", []):
            parent_group_id = method_or_create(
                StaffGroup, ancestor, parent_id=parent_group_id
            )

        return parent_group_id


class UserMixin:
    """
    Миксин для создания базовых профилей пользователей
    """

    def get_or_create_user(self, user_data: dict):
        cache_key = f"{user_data['uid']}:{user_data['login']}"
        cached_users = self.cache.get("users")

        if not cached_users.get(cache_key):
            user_kwargs = {
                User.YAUID_FIELD: user_data["uid"],
                User.USERNAME_FIELD: user_data["login"],
            }
            user = User.objects.filter(**user_kwargs).first()
            if user is None:
                user = User.objects.create(**user_kwargs)
                self.counts["user_created"] += 1

            cached_users[cache_key] = user
            self.counts["users_size"] += 1

        else:
            self.counts["users_hits"] += 1

        return cached_users[cache_key]


class StaffOfficeLoader(ModelMixin, BaseLoader):
    """
    Загрузчик всех офисов (а также городов и стран)
    """

    endpoint = "offices"
    fields = [
        "id",
        "name",
        "code",
        "is_deleted",
        {"city": ("id", "name", "is_deleted")},
        {"city.country": ("id", "name", "code", "is_deleted")},
    ]

    def update(self, **kwargs) -> None:
        self.counts["created"] = 0

        for office in self.get_items():
            city = office.get("city", {})
            country = city.get("country", {})

            self.update_or_create_model(StaffCountry, country)
            self.update_or_create_model(StaffCity, city, country_id=country["id"])
            self.update_or_create_model(
                StaffOffice, office, city_id=city["id"], no_cache=True
            )

            self.counts["processed"] += 1

        log.info("Summary: %s", self.counts)
        self.clear_cache()


class StaffGroupLoader(GroupMixin, ModelMixin, BaseLoader):
    """
    Загрузчик подразделений со Стаффа (включая родительские)
    """

    endpoint = "groups"
    fields = [
        "id",
        "name",
        "type",
        "level",
        "url",
        "is_deleted",
        {"ancestors": ("id", "type", "name", "level", "is_deleted")},
    ]
    init_kwargs = {
        "type": "department",
    }

    def __init__(self, group_ids: Optional[Union[list, int]] = None, **kwargs):
        if group_ids and isinstance(group_ids, int):
            group_ids = [group_ids]
        self.group_ids = group_ids

        super().__init__(**kwargs)

    def get_init_kwargs(self) -> dict:
        kwargs = super().get_init_kwargs()
        if self.group_ids:
            kwargs["id"] = ",".join(map(str, self.group_ids))
        return kwargs

    def update(self, **kwargs) -> None:
        self.counts["created"] = 0

        for group in self.get_items(**kwargs):
            self.update_or_create_group(group)
            self.counts["processed"] += 1

        log.info("Summary: %s", self.counts)
        self.clear_cache()


class StaffProfileProfileLoader(UserMixin, ModelMixin, BaseProfileLoader):
    """
    Загрузчик данных профиля со Стаффа
    """

    endpoint = "persons"
    uid_field = "uid"
    fields = [
        "_meta",
        "uid",
        "name",
        "login",
        "is_deleted",
        {"chief": ("uid", "login")},
        {"location.office": ("id", "code", "name", "is_deleted", "city")},
        {"official": ("join_at", "is_dismissed", "position")},
        "language.native",
    ]

    def update_profile(self, profile: StaffProfile, data: dict):
        log.debug("%s: update profile: %s with %r", __class__.__name__, profile, data)
        profile.is_active = not data.get("is_deleted", False)

        official = data.get("official")
        if official:
            join_at = official.get("join_at")
            if join_at:
                try:
                    join_at = datetime.strptime(join_at, "%Y-%m-%d").replace(
                        tzinfo=timezone.utc
                    )
                    profile.joined_at = join_at
                except ValueError:
                    pass

            if official.get("position"):
                profile.position_ru = lookup_field(official, "position.ru")
                profile.position_en = lookup_field(official, "position.en")

            profile.is_dismissed = official.get("is_dismissed", False)

        office = lookup_field(data, "location.office")
        if office:
            city = office.get("city", {})
            country = city.get("country", {})

            self.get_or_create_model(StaffCountry, country)
            self.get_or_create_model(StaffCity, city, country_id=country["id"])
            self.get_or_create_model(StaffOffice, office, city_id=city["id"])

            profile.office_id = office["id"]

        if data.get("name"):
            profile.first_name_ru = lookup_field(data, "name.first.ru")
            profile.last_name_ru = lookup_field(data, "name.last.ru")
            profile.first_name_en = lookup_field(data, "name.first.en")
            profile.last_name_en = lookup_field(data, "name.last.en")

        language_native = lookup_field(data, "language.native")
        if language_native:
            profile.language_native = language_native

        profile.save()


class StaffGroupProfileLoader(GroupMixin, ModelMixin, BaseProfileLoader):
    """
    Загрузчик данных о группах пользователя со Стаффа
    """

    endpoint = "groupmembership"
    uid_field = "person.uid"
    init_kwargs = {
        "group.type": "department",
    }
    fields = [
        {"person": ("uid", "login", "is_deleted")},
        {
            "group": ("id", "type", "name", "level", "is_deleted"),
            "group.ancestors": ("id", "type", "name", "level", "is_deleted"),
        },
    ]

    def update(self, **kwargs):
        cached_profiles = self.cache.get("profiles", cast=set)

        for profile, userdata in self.profile_data():
            group = userdata.get("group")
            if not group:
                continue

            if not getattr(profile, "_staff_groups_pks", None):
                profile._staff_groups_pks = set()

            self.get_or_create_group(group)

            profile._staff_groups_pks.add(group["id"])
            cached_profiles.add(profile)

        for profile in cached_profiles:
            profile.groups.set(list(profile._staff_groups_pks))
            self.counts["profiles_processed"] += 1

        log.info("Summary: %s", self.counts)
        self.clear_cache()


class StaffLeadershipProfileLoader(GroupMixin, ModelMixin, BaseProfileLoader):
    """
    Загрузчик данных о руководстве в группах на Стаффе
    """

    endpoint = "departmentstaff"
    uid_field = "person.uid"
    init_kwargs = {
        "department_group.type": "department",
    }
    fields = [
        {"person": ("uid", "login", "is_deleted")},
        {"department_group": ("id", "name", "level", "type", "is_deleted")},
        "role",
    ]

    def update(self, **kwargs):
        cached_profiles = self.cache.get("profiles", cast=set)

        for profile, userdata in self.profile_data():
            group = userdata.get("department_group", {})
            if not group:
                continue

            self.get_or_create_group(group)

            staff_leadership = self.get_or_create_leadership(
                profile=profile, group_id=group["id"], role=userdata.get("role")
            )

            if not getattr(profile, "_staff_leadership_pks", None):
                profile._staff_leadership_pks = set()

            profile._staff_leadership_pks.add(staff_leadership.pk)
            cached_profiles.add(profile)

        # деактивируем членство в остальных группах
        for profile in cached_profiles:
            StaffLeadership.objects.filter(profile=profile).exclude(
                pk__in=profile._staff_leadership_pks
            ).update(is_active=False)

            self.counts["profiles_processed"] += 1

        log.info("Summary: %s", self.counts)
        self.clear_cache()

    def get_or_create_leadership(self, **kwargs) -> "StaffLeadership":
        staff_leadership, created = StaffLeadership.objects.get_or_create(**kwargs)

        if not created and not staff_leadership.is_active:
            staff_leadership.is_active = True
            staff_leadership.save()

        if created:
            self.counts["leaderships_created"] += 1

        return staff_leadership


class StaffDismissedProfileLoader(StaffProfileProfileLoader):
    """
    Загрузчик данных уволенных сотрудников

    (частичное обновление профиля пользователя)
    """

    endpoint = "persons"
    fields = [
        "_meta",
        "uid",
        "is_deleted",
        "official.is_dismissed",
    ]
