import logging
from datetime import timedelta

from django.core.cache import cache
from django.db import transaction
from django.utils import timezone

from mentor.contrib.staff.settings import (
    STAFF_PROFILE_INACTIVE_LIMIT,
    STAFF_PROFILE_INACTIVE_TIME,
    STAFF_PROFILE_REFRESH_TIME,
    STAFF_PROFILE_UPDATED_CACHE_KEY,
)

from .loaders import (
    StaffDismissedProfileLoader,
    StaffGroupLoader,
    StaffGroupProfileLoader,
    StaffLeadershipProfileLoader,
    StaffOfficeLoader,
    StaffProfileProfileLoader,
)
from .models import StaffProfile

log = logging.getLogger(__name__)


def is_profile_updated(user_pk: int):
    return cache.get(STAFF_PROFILE_UPDATED_CACHE_KEY.format(user_pk), False)


def set_profile_updated(user_pk: int, timeout=None):
    if timeout is None:
        timeout = STAFF_PROFILE_REFRESH_TIME
    cache.set(
        STAFF_PROFILE_UPDATED_CACHE_KEY.format(user_pk), True, timeout=int(timeout)
    )


def update_inactive_profiles() -> None:
    inactive_time = timezone.now() - timedelta(seconds=STAFF_PROFILE_INACTIVE_TIME)
    queryset = StaffProfile.objects.filter(
        modified__lt=inactive_time, user__yauid__isnull=False
    )

    if not queryset.exists():
        log.info("No inactive profiles found")
        return

    uids = list(
        queryset.values_list("user__yauid", flat=True)[:STAFF_PROFILE_INACTIVE_LIMIT]
    )
    StaffDismissedProfileLoader(uids).update()


def load_staff_profile(user_pk: int) -> bool:
    """
    Загружает профиль пользователя со стаффа

    При загрузке проверяет актуальность профиля по полю modified,
    если не выставлен параметр force_load=True

    :param user_pk: ID пользователя
    :return: True при успешной загрузке, False - при отмене загрузки
    """
    staff_profile = StaffProfile.objects.filter(
        user_id=user_pk, user__yauid__isnull=False
    ).first()

    if not staff_profile:
        log.info("No profile found: %d", user_pk)
        return False

    loaders = [
        StaffProfileProfileLoader,
        StaffGroupProfileLoader,
        StaffLeadershipProfileLoader,
    ]

    with transaction.atomic():
        for loader in loaders:
            loader(staff_profile.uid).update()

    set_profile_updated(user_pk)

    log.info("Profile %d was updated", user_pk)

    return True


def load_staff_offices(**kwargs):
    StaffOfficeLoader(**kwargs).update()


def load_staff_groups(**kwargs):
    StaffGroupLoader(**kwargs).update()
