from typing import Generator, Iterable

from django.utils.translation import get_language as _get_language

from mentor.contrib.staff import settings


def get_language():
    lang = _get_language()

    if lang is None:
        return settings.STAFF_DEFAULT_LANGUAGE

    if "-" in lang:
        lang = lang.split("-")[0]

    return lang


def parse_fields(fields: Iterable, prefix: str = "") -> Generator[str, None, None]:
    """
    Разворачивает словари в проский список. Возвращает генератор

    Например:
    fields = [
        'id',
        'name',
        {'group': ['id', name']},
        {'group.ancestors': ['name', 'level']},
    ]

    будет преобразован в:
    fields = [
        'id',
        'name',
        'group.id',
        'group.name',
        'group.ancestors.name',
        'group.ancestors.level',
    ]

    :param fields: список данных
    :param prefix: префикс (используется при вычислениях)
    :return:
    """
    if prefix:
        prefix += "."

    for item in fields:
        if isinstance(item, str):
            yield f"{prefix}{item}"
        elif isinstance(item, dict):
            for key, vals in item.items():
                yield from parse_fields(vals, prefix=f"{prefix}{key}")
        else:
            raise ValueError(f"Unknown field: {item}")


def lookup_field(item, field, default=None):
    """
    Возвращает вложенные данные из словарей

    Например:
    data = {
        'location': {
            'office': {
                'name': 'red',
            }
        }
    }

    lookup_field(data, 'location.office.name')  # возвращает 'red'
    lookup_field(data, 'location.city.test', 'unknown')  # возвращает 'unknown'

    :param item: словарь
    :param field: путь до нужного поля
    :param default: значение по умолчанию
    :return:
    """
    keys = field.split(".")
    value = item
    for k in keys:
        value = value.get(k, None) if isinstance(value, dict) else None
        if value is None:
            return default

    return value
