from django.contrib.admin import ModelAdmin
from django.contrib.auth import get_user_model
from django.core.urlresolvers import reverse
from django.utils.html import format_html_join
from django.utils.translation import ugettext_lazy as _

User = get_user_model()


class ParentProfileAdmin(ModelAdmin):
    """
    Профили учителей
    """
    list_display = (
        'parent',
        'get_parent_name',
        'get_children',
    )
    search_fields = (
        'parent__username',
        'parent__first_name',
        'parent__middle_name',
        'parent__last_name',
        'children__username',
        'children__first_name',
        'children__middle_name',
        'children__last_name',
    )

    USER_ADMIN_CHANGE_REVERSE = 'admin:%s_%s_change' % (User._meta.app_label, User._meta.model_name)

    def get_queryset(self, request):
        """
        Добавляет аккаунт родители и аккаунты его учеников к запросу
        """
        return (super(ParentProfileAdmin, self).get_queryset(request)
                .select_related('parent')
                .prefetch_related('children'))

    def get_parent_name(self, obj):
        """
        Возвращает полное имя родителя
        """
        return obj.parent.get_full_name()

    get_parent_name.short_description = _('ФИО')
    get_parent_name.admin_order_field = 'parent__last_name'

    def get_children(self, obj):
        """
        Возвращает список детей. Для каждого ребенка пишется полное имя и имя
        пользователя в скобках. Имя пользователя также ссылается на страницу
        ученика в админке
        """
        return format_html_join(
            '',
            '<p>{0} (<a href="{1}" target="_blank">{2}</a>)</p>',
            ((child.get_full_name(),
              reverse(self.USER_ADMIN_CHANGE_REVERSE, args=(child.id,)),
              child.username) for child in obj.children.all())
        )

    get_children.short_description = _('Дети')
