from django_object_actions import DjangoObjectActions

from django.conf import settings
from django.conf.urls import url
from django.contrib import admin
from django.contrib.auth import get_user_model
from django.contrib.auth.admin import UserAdmin as BaseUserAdmin
from django.db.models.expressions import Q
from django.http import HttpResponseRedirect
from django.urls import reverse
from django.utils.translation import ugettext_lazy as _

from kelvin.accounts.admin.autocomplete import UsersAutocomplete
from kelvin.accounts.admin.forms import UserForm
from kelvin.achievery.admin import UserAchievementInlineAdmin
from kelvin.common.admin import build_dropdown_filter_with_title
from kelvin.courses.models import Course, CourseStudent
from kelvin.idm.admin import UserIDMRoleRequestInlineAdmin
from kelvin.mail.models import ScheduledEmail

from ...utils import invalidate_user_courses_cache

User = get_user_model()


class StudentCoursesInline(admin.TabularInline):
    model = CourseStudent
    fields = ('course_name',)
    readonly_fields = fields
    extra = 0
    max_num = 0
    can_delete = False
    verbose_name_plural = _(u"курсы ученика")

    def course_name(self, obj):
        return obj.course.name

    course_name.short_description = _(u"название учебного курса")

    def get_queryset(self, request):
        queryset = super(StudentCoursesInline, self).get_queryset(request)
        return queryset.select_related('course')

    def has_add_permission(self, request):
        return False


class TeacherCoursesInline(admin.TabularInline):
    model = Course
    fields = ('name',)
    readonly_fields = fields
    extra = 0
    max_num = 0
    fk_name = 'owner'
    can_delete = False
    verbose_name_plural = _(u"преподаватель / контент-менеджер на курсах")

    def has_add_permission(self, request):
        return False


class EduUserAdmin(DjangoObjectActions, BaseUserAdmin):
    list_display = (
        'id',
        'username',
        'get_full_name',
        'date_joined',
    )
    list_filter = (
        'is_content_manager',
        'is_active',
        'is_staff',
        'is_teacher',
    )
    search_fields = ['username', 'email', 'first_name', 'last_name']
    date_hierarchy = 'date_joined'
    list_display_links = ('id', 'username')
    ordering = ['-date_joined', 'username']

    form = UserForm

    fieldsets = (
        (None, {
            'fields': (
                'username',
                'email',
                'first_name',
                'last_name',
                'middle_name',
                'gender',
                'unsubscribed',
                'is_tos_accepted',
                'is_staff',
                'is_superuser',
                'is_active',
                'is_content_manager',
                'is_teacher',
                'phone_number',
                'default_avatar_id',
                'display_name',
                'parent_code',
                'yauid',
                'experiment_flags',
                'is_dismissed',
            ),
        }),
        ('Права пользователя', {
            'classes': ('collapse',),
            'fields': ('groups', 'user_permissions'),
        }),
    )
    add_fieldsets = [
        (None, {
            'classes': ['wide'],
            'fields': ['username', 'email', 'first_name', 'last_name']
        }),
    ]

    inlines = [
        StudentCoursesInline,
        TeacherCoursesInline,
    ]

    if settings.ACHIEVERY_ENABLED:
        inlines += [
            UserAchievementInlineAdmin,
        ]

    if settings.IDM_ENABLED:
        inlines += [
            UserIDMRoleRequestInlineAdmin,
        ]

    actions = [
        'send_mail_to_users',
    ]

    change_actions = [
        'update_staff_data',
        'update_user_course_cache',
    ]

    def update_staff_data(self, request, obj):
        from kelvin.tags.tasks import (
            tag_users_from_staff_groups_by_ids,
            tag_users_from_staff_location_by_ids,
            tag_users_from_staff_role_by_ids,
        )

        tag_users_from_staff_groups_by_ids.delay(user_ids=[obj.id])
        tag_users_from_staff_location_by_ids.delay(user_ids=[obj.id])
        tag_users_from_staff_role_by_ids.delay(user_ids=[obj.id], role='chief')
        tag_users_from_staff_role_by_ids.delay(user_ids=[obj.id], role='hr_partner')

        self.message_user(request, _("Staff data for user '{}' will be updated soon.".format(obj.username)))

    update_staff_data.label = _("Обновить данные со Стаффа")

    def update_user_course_cache(self, request, obj):
        from kelvin.accounts.update_user_courses_task import set_update_user_courses_task

        invalidate_user_courses_cache(obj)
        set_update_user_courses_task(user_id=obj.id, force_update=True)

        self.message_user(request, _("Кэш курсов пользователя '{}' скоро будет обновлен.".format(obj.username)))

    update_user_course_cache.label = _("Обновить кэш курсов")

    def has_add_permission(self, request):
        # пользователи добавляются через яндексовую авторизацию
        return False

    def get_urls(self):
        """
        Добавляет вьюхи для импорта и автокомплита пользователей
        """
        info = self.model._meta.app_label, self.model._meta.model_name
        autocomplete_url = url(
            r'autocomplete/$',
            self.admin_site.admin_view(UsersAutocomplete.as_view()),
            name='{0}_{1}_autocomplete'.format(*info),
        )
        return [
            autocomplete_url,
        ] + super().get_urls()

    def get_user_subscriptions(self, user):
        return {}

    def send_mail_to_users(modeladmin, request, queryset):
        """
        Отправка рассылки выбранным пользователям.

        Записывает в сессию адреса выбранных пользователей и отправляет на форму
        отправки рассылок.
        """
        emails = queryset.filter(
            email__isnull=False
        ).values_list(
            'email',
            flat=True,
        )
        emails = set(emails)

        emails_event = queryset.filter(
            eventregistration__email__isnull=False
        ).values_list('eventregistration__email', flat=True)
        emails.update(emails_event)

        request.session['mail_sender__emails'] = list(emails)

        return HttpResponseRedirect(
            reverse('admin:mail_emailtemplate_mail_sender'))

    send_mail_to_users.short_description = _('Отправить рассылку')

    def render_change_form(self, request, context, **kwargs):
        """
        Добавляет в контекст рендеринга курсы и группы ученика и
        учителя/контент-менеджера
        """
        user = context.get('original')
        if user:
            # Подписки пользователя
            mail_subscription_expression = (
                Q(group__in=user.groups.all()) |
                (Q(course__copy_of__isnull=True) &
                 Q(course__copies__students=user)) |
                Q(course__students=user)
            )

            if user.email:
                mail_subscription_expression |= Q(
                    email_list__icontains=user.email)

            user_subscriptions = ScheduledEmail.objects.filter(
                mail_subscription_expression
            ).order_by('id').distinct('id')

            context.update({'user_subscriptions': user_subscriptions})

            # children / parents
            if user.is_parent and hasattr(user, 'parent_profile'):
                context.update({
                    'children': user.parent_profile.children.all()
                })

            context.update({
                'parents': self.model.objects.filter(parent_profile__children=user)
            })

        return super(EduUserAdmin, self).render_change_form(
            request, context, **kwargs
        )


class UserContentManagerAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'username',
        'first_name',
        'last_name',
        'is_content_manager',
        'is_teacher',
    )
    readonly_fields = (
        'id',
        'username',
        'first_name',
        'last_name',
    )
    fields = readonly_fields + (
        'is_content_manager',
        'is_teacher',
    )
    search_fields = ['username', 'email', 'first_name', 'last_name']


class UserProjectAdmin(admin.ModelAdmin):
    raw_id_fields = ('user',)
    list_filter = [
        'project',
        (
            'user__username',
            build_dropdown_filter_with_title('Студенты'),
        ),
    ]
    list_select_related = (
        'project',
        'user',
    )
