import logging
from builtins import object

from django.conf import settings
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.tags.models import Tag, TagTypeStudyGroup

logger = logging.getLogger(__name__)


@python_2_unicode_compatible
class StudyGroup(models.Model):
    """Учебная группа."""
    owner = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Владелец курса'),
        on_delete=models.CASCADE,
    )

    students = models.ManyToManyField(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Студенты группы'),
        through='StudyGroupUserLink',
        related_name='study_groups',
        blank=True,
    )

    project = models.ForeignKey(
        'projects.Project',
        verbose_name=_('Проект группы'),
        null=True,
        on_delete=models.SET_NULL,
    )

    slug = models.SlugField(
        verbose_name=_('Слаг-идентификатор группы'),
        max_length=255,
        unique=True,
    )

    name = models.CharField(
        verbose_name=_('Название группы'),
        max_length=255,
    )

    class Meta(object):
        verbose_name = _('Учебная группа')
        verbose_name_plural = _('Учебные группы')
        permissions = [
            ("can_manage_groups", "Can manage all study groups"),
        ]

    def __str__(self):
        return 'ID: {0} Slug: {1}, Owner: {2}'.format(
            self.id, self.slug, self.owner)

    def save(self, *args, **kwargs):
        Tag.objects.get_or_create(
            project=self.project,
            value=self.slug,
            type=TagTypeStudyGroup.get_db_type(),
        )
        return super(StudyGroup, self).save(*args, **kwargs)


@python_2_unicode_compatible
class StudyGroupUserLink(models.Model):
    """
    Связь учебной группы с учеником
    """

    user = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Ученик'),
        on_delete=models.CASCADE,
    )
    study_group = models.ForeignKey(
        StudyGroup,
        verbose_name=_('Учебная группа'),
        on_delete=models.CASCADE,
    )

    class Meta(object):
        verbose_name = _('Связь учебной группы с учеником')
        verbose_name_plural = _('Связи учебных групп с учениками')
        unique_together = (("user", "study_group"),)

    def __str__(self):
        return 'StudyGroup {0}, User {1}, ID: {2}'.format(
            self.study_group, self.user, self.id)
