import logging
from builtins import object, range

from model_utils import FieldTracker

from django.conf import settings
from django.contrib.auth.models import PermissionsMixin, UserManager
from django.db import models
from django.utils.crypto import get_random_string
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.templatetags.common_tags import Genders
from kelvin.common.utils import safe_filename

logger = logging.getLogger(__name__)


def get_avatar_upload_path(instance, filename):
    return safe_filename(filename, 'avatar')


class AbstractUser(PermissionsMixin, models.Model):
    username = models.CharField(
        _('username'), max_length=255, unique=True, blank=True,
        null=False,
        error_messages={
            'unique': _("A user with that username already exists."),
        })
    is_staff = models.BooleanField(
        _('staff status'),
        default=False,
        help_text=_('Designates whether the user can log into this admin '
                    'site.'))
    is_active = models.BooleanField(
        _('active'),
        default=True,
        help_text=_('Designates whether this user should be treated '
                    'as active. Unselect this instead of deleting '
                    'accounts.'))
    date_joined = models.DateTimeField(_('date joined'),
                                       auto_now_add=True)

    last_login = models.DateTimeField(_('last login'), blank=True, null=True)

    USERNAME_FIELD = 'username'
    REQUIRED_FIELDS = []

    objects = UserManager()

    # from AbstractBaseUser
    def get_username(self):
        """Return the identifying username for this User"""
        return getattr(self, self.USERNAME_FIELD)

    def __str__(self):
        return self.get_username()

    def natural_key(self):
        return (
            self.get_username(),
        )

    def set_unusable_password(self):
        # используется в `django_yauth`
        pass

    @property
    def has_usable_password(self):
        # используется в `django_yauth`
        return False

    @property
    def is_authenticated(self):
        # аутентификация в `request.yauser`
        return True

    @property
    def is_anonymous(self):
        return False

    def get_short_name(self):
        """Return short user name"""
        return self.username

    def can_add_related(self):
        """
        Return a decision about whether the user can add a new item.
        Need for admin widget only.
        """
        return True

    class Meta(object):
        abstract = True
        verbose_name = _('Пользователь')
        verbose_name_plural = _('Пользователи')


class GenderMixin(models.Model):
    """
    Mixin with functionality connected with user gender.
    """
    gender = models.IntegerField(
        _('Пол'),
        choices=[
            (value, key) for key, value in list(Genders.GENDERS_DICT.items())
        ],
        default=Genders.GENDER_UNSPECIFIED,
        blank=True,
    )

    def save(self, *args, **kwargs):
        if not self.pk and (
            self.gender not in list(Genders.GENDERS_DICT.values())
        ):
            # User from Ya.Passport may have other values as Gender
            self.gender = Genders.GENDER_UNSPECIFIED
        super(GenderMixin, self).save(*args, **kwargs)

    class Meta(object):
        abstract = True


class PersonalDataMixin(models.Model):
    MAX_UNDERAGE_USERNAME_LENGTH = 18

    email = models.EmailField(
        _("E-mail"),
        max_length=255,
        unique=True,
        null=True,
        blank=True,
        error_messages={
            'unique': _("A user with that email already exists."),
        }
    )

    first_name = models.CharField(_('Имя'), max_length=255, blank=True, null=True)
    last_name = models.CharField(_('Фамилия'), max_length=255, blank=True, null=True)
    middle_name = models.CharField(_('Отчество'), max_length=255, blank=True, null=True)
    phone_number = models.CharField(_('Номер телефона'), max_length=31, blank=True)

    avatar = models.ImageField(
        verbose_name=_('Аватарка'),
        blank=True,
        null=True,
        upload_to=get_avatar_upload_path,
    )

    default_avatar_id = models.CharField(_('ID аватарки'), max_length=255, blank=True)
    display_name = models.CharField(_('Отображаемое имя'), max_length=255, blank=True)

    def get_full_name(self):
        """
        Returns full name with last name and first name or username
        """
        if self.first_name and self.last_name:
            if self.middle_name:
                full_name = '{} {} {}'.format(self.first_name, self.middle_name, self.last_name)

            full_name = '{} {}'.format(self.first_name, self.last_name)
        else:
            full_name = self.get_username()

        return full_name

    get_full_name.short_description = 'ФИО'
    get_full_name.admin_order_field = 'last_name'

    @property
    def user_label(self):
        return '{}: {} ({})'.format(self.username, self.get_full_name(), self.id)

    def save(self, *args, **kwargs):
        self.email = self.email or None
        super(PersonalDataMixin, self).save(*args, **kwargs)

    class Meta(object):
        abstract = True


class LegalMixin(models.Model):
    is_tos_accepted = models.BooleanField(
        _('Согласен с правилами пользования сервисом'),
        default=False,
        help_text=_('Ученик принял правила пользования сервисом'),
    )

    class Meta(object):
        abstract = True


class SubscriptionsMixin(models.Model):
    """
    Mixin with functionality connected subscriptions
    """
    unsubscribed = models.BooleanField(
        verbose_name=_('Отписан от рассылок'),
        blank=True,
        default=False,
    )

    class Meta(object):
        abstract = True


class ContentManagerMixin(models.Model):
    """
    Mixin with functionality connected with content manager.
    """
    is_content_manager = models.BooleanField(
        _('Контент-менеджер'),
        default=False,
    )

    class Meta(object):
        abstract = True


class TeacherMixin(models.Model):
    is_teacher = models.BooleanField(
        verbose_name=_('Учитель'),
        default=False,
    )

    class Meta(object):
        abstract = True


class ParentMixin(models.Model):
    CODE_LENGTH = 6  # минимальная длина кода
    CODE_LENGTH_MAX = 12  # длина кода строго меньше этого значения
    CODE_CREATION_ATTEMPTS = 4  # количество попыток создания кода
    CODE_FIELD_LENGTH = 15
    CODE_ALLOWED_CHARS = '0123456789ABCDEFGHKLMNPRSTUVWXYZ'

    parent_code = models.CharField(
        _('Код для родителя'),
        max_length=CODE_FIELD_LENGTH,
        unique=True,
        blank=True,
        db_index=True,
    )
    is_parent = models.BooleanField(
        _('Родитель'),
        default=False,
    )

    @classmethod
    def _generate_code(cls, length):
        return get_random_string(length, allowed_chars=cls.CODE_ALLOWED_CHARS)

    @classmethod
    def generate_parent_code(cls):
        for length in range(cls.CODE_LENGTH, cls.CODE_LENGTH_MAX):
            for _ in range(cls.CODE_CREATION_ATTEMPTS):
                code = cls._generate_code(length)
                if not cls.objects.filter(parent_code=code).exists():
                    return code
        logger.warning('Short parent code not created, generated long code')
        return cls._generate_code(cls.CODE_FIELD_LENGTH)

    def save(self, *args, **kwargs):
        self.parent_code = self.parent_code or self.generate_parent_code()
        super(ParentMixin, self).save(*args, **kwargs)

    class Meta(object):
        abstract = True


class StudentMixin(object):
    @property
    def is_student(self):
        return self.coursestudent_set.exists()


class SupportMixin:
    @cached_property
    def is_support(self):
        return self.groups.filter(name=settings.SUPPORT_GROUP).exists()


class User(
    AbstractUser,
    GenderMixin,
    PersonalDataMixin,
    LegalMixin,
    SubscriptionsMixin,
    ContentManagerMixin,
    TeacherMixin,
    ParentMixin,
    StudentMixin,
    SupportMixin,
):
    yauid = models.BigIntegerField(
        verbose_name=_('Яндекс UID пользователя'),
        null=True,
        blank=True,
        db_index=True,
    )

    experiment_flags = JSONField(
        verbose_name=_('Флаги для экспериментов'),
        blank=False,
        default={},
    )

    #  Поле is_curator автовычисляется в сигналах сушности CoursePermission
    is_curator = models.BooleanField(
        verbose_name=_('Куратор'),
        default=False,
        help_text=_('Если пользователь куратор хотя бы в одном курсе, то здесь True')
    )

    is_dismissed = models.BooleanField(
        verbose_name=_('Бывший сотрудник'),
        default=False,
    )

    tracker = FieldTracker(fields=['is_teacher', 'is_parent'])


class UserContentManager(User):
    class Meta(User.Meta):
        verbose_name = _('Пользователь-контент менеджер')
        verbose_name_plural = _('Пользователи-контент менеджеры')
        proxy = True


class UserStaffGroups(models.Model):
    user = models.OneToOneField(
        User,
        verbose_name=_('Пользователь'),
        unique=True,
        db_index=True,
    )
    staff_groups = models.CharField(
        verbose_name=_('Staff-группы'),
        help_text=_('Отсортированные по level-у группы'),
        max_length=4096,
        blank=False,
        null=False
    )
    date_updated = models.DateTimeField(
        auto_now=True,
        verbose_name=_('Дата обновления записи')
    )
