from builtins import object

from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.courses.tasks import add_mandatory_courses_to_single_student

from ..utils import add_project_to_users_cache, del_project_from_users_cache


@python_2_unicode_compatible
class UserProject(models.Model):
    """
        Сущность для связи пользователя и проекта
        Выделено в отдельный класс вместо ManyToManyRelationship-поля
        для тото, чтобы иметь возможность связи
        пользователь-проект добавлять дополнительную информацию -
        - например, роли/права пользователя в данном проекте
    """
    user = models.ForeignKey(
        'accounts.User',
        null=False,
        blank=False,
        verbose_name=_("Пользователь")
    )
    project = models.ForeignKey(
        'projects.Project',
        null=False,
        blank=False,
        verbose_name=_("Проект")
    )
    nda_accepted = models.BooleanField(
        verbose_name=_('NDA принят'),
        default=False,
    )

    def update_user_experiment_flags(self):
        """Напонлнить пользовательские флаги экспериментов дефолтными для этого проекта."""
        self.user.experiment_flags.update(self.project.default_experiment_flags)
        self.user.save()

    def assign_mandatory_courses(self):
        """Добавить пользователю все обязательные курсы для этого проекта."""
        add_mandatory_courses_to_single_student.delay(self.user.id)
        add_project_to_users_cache(self)

    def save(self, *args, **kwargs):
        """
        При добавлении пользователя в проект:
             1. Пытаемся добавить ему все обязательные курсы.
             2. Напонлнить пользовательские флаги экспериментов дефолтными для этого проекта.
        """
        is_new_object = self.pk is None
        super(UserProject, self).save(*args, **kwargs)
        if is_new_object:
            self.assign_mandatory_courses()
            self.update_user_experiment_flags()

    def delete(self, *args, **kwargs):
        del_project_from_users_cache(self)
        super(UserProject, self).delete(*args, **kwargs)

    def __str__(self):
        return "User: {}, Project: {}".format(self.user.id, self.project.id)

    class Meta(object):
        verbose_name = _('Проект пользователя')
        verbose_name_plural = _('Проекты пользователей')
        unique_together = ('user', 'project')
