from django.conf import settings

from rest_framework.permissions import SAFE_METHODS, BasePermission

from kelvin.common.staff_reader import staff_reader
from kelvin.courses.models import Course, CourseLessonLink


class IsSuperuser(BasePermission):
    """
    Проверка, что пользователь является суперпользователем
    """
    def has_permission(self, request, view):
        return getattr(request.user, 'is_superuser', False)


class IsStaff(BasePermission):
    """
    Проверка, что пользователь является админом
    """
    def has_permission(self, request, view):
        return getattr(request.user, 'is_staff', False)


class IsTeacher(BasePermission):
    """
    Проверка, что пользователь является учителем
    """
    def has_permission(self, request, view):
        return getattr(request.user, 'is_teacher', False)


class IsContentManager(BasePermission):
    """
    Доступ только контент-менеджерам
    """
    def has_permission(self, request, view):
        return getattr(request.user, 'is_content_manager', False)


class IsParent(BasePermission):
    """
    Доступ только родителям
    """
    def has_permission(self, request, view):
        return getattr(request.user, 'is_parent', False)


class IsStaffChief(BasePermission):

    def has_permission(self, request, view):
        return staff_reader.get_user_roles(
            request.user.username,
        ).get('chief', False)


class IsStaffHRBP(BasePermission):

    def has_permission(self, request, view):
        return staff_reader.has_department_role(
            username=request.user.username,
            role='hr_partner',
        )


class IsCurator(BasePermission):

    def has_permission(self, request, view):
        return getattr(request.user, 'is_curator', False)


class IsSupport(BasePermission):
    def has_permission(self, request, view):
        return request.user and request.user.is_support


class ObjectStudyGroupOwner(BasePermission):

    def has_object_permission(self, request, view, obj):
        if not request.user:
            return False

        return (
            obj.owner == request.user or
            request.user.has_perm('courses.can_edit_course_in_lab')
        )


class ObjectCanManageStudyGroup(BasePermission):

    def has_object_permission(self, request, view, obj):
        return request.user.has_perm('accounts.can_manage_groups')


class ObjectOwnerOrContentManager(BasePermission):
    """
    Проверка, что пользователь является владельцем объекта
    или контент-менеджером
    """
    def has_object_permission(self, request, view, obj):
        if not request.user:
            return False
        return (
            request.user.is_content_manager or
            request.user.id == obj.owner_id
        )

        if isinstance(obj, Course):
            return (
                request.user.has_perm('courses.can_edit_course_in_lab') and
                obj.available_for_support
            )

        if isinstance(obj, CourseLessonLink):
            return (
                request.user.has_perm('courses.can_edit_course_in_lab') and
                obj.course.available_for_support
            )

        return False


class ObjectReadOnly(BasePermission):
    """
    Проверка, что запрос на чтение
    """
    def has_object_permission(self, request, view, obj):
        return request.method in SAFE_METHODS


class ObjectForAuthenticated(BasePermission):
    """
    Проверка, что пользователь авторизован
    """
    def has_object_permission(self, request, view, obj):
        return request.user.is_authenticated


class ObjectForTeacher(BasePermission):
    """
    Проверка, что пользователь является учителем
    """
    def has_object_permission(self, request, view, obj):
        return getattr(request.user, 'is_teacher', False)


class IsStaff(BasePermission):
    """
    Проверка, что пользователь является админом
    """
    def has_permission(self, request, view):
        return getattr(request.user, 'is_staff', False)


class IsContentManager(BasePermission):
    """
    Доступ только контент-менеджерам
    """
    def has_permission(self, request, view):
        return getattr(request.user, 'is_content_manager', False)


class ReadOnly(BasePermission):
    """
    Проверка, что запрос на чтение
    """
    def has_permission(self, request, view):
        return request.method in SAFE_METHODS


class ObjectForOwner(BasePermission):
    """
    Проверка, что пользователь является владельцем объекта
    """
    def has_object_permission(self, request, view, obj):
        if not request.user:
            return False

        if request.user.id == obj.owner_id:
            return True

        if isinstance(obj, Course):
            return (
                request.user.has_perm('courses.can_edit_course_in_lab') and
                obj.available_for_support
            )

        if isinstance(obj, CourseLessonLink):
            return (
                request.user.has_perm('courses.can_edit_course_in_lab') and
                obj.course.available_for_support
            )

        return False


class ObjectForContentManager(BasePermission):
    """
    Доступ только контент-менеджерам
    """
    def has_permission(self, request, view):
        return False

    def has_object_permission(self, request, view, obj):
        if getattr(request.user, 'is_content_manager', False):
            return True

        if isinstance(obj, Course):
            return (
                request.user.has_perm('courses.can_edit_course_in_lab') and
                obj.available_for_support
            )

        if isinstance(obj, CourseLessonLink):
            return (
                request.user.has_perm('courses.can_edit_course_in_lab') and
                obj.course.available_for_support
            )

        return False


class ObjectForStaff(BasePermission):
    """
    Проверка, что пользователь является админом
    """
    def has_object_permission(self, request, view, obj):
        return getattr(request.user, 'is_staff', False)


class ObjectForAuthenticated(BasePermission):
    """
    Проверка, что пользователь авторизован
    """
    def has_object_permission(self, request, view, obj):
        return request.user.is_authenticated


class ObjectSelf(BasePermission):
    """
    Доступ пользователя только к своему профлю
    """
    def has_object_permission(self, request, view, obj):
        return request.user.id == obj.id


class ObjectIsUserInCourseOwner(BasePermission):
    """
        Доступ к профилю человека, если он ученик в курсе, которым мы владеем
        """

    def has_object_permission(self, request, view, obj):
        return Course.objects.filter(
            students__id=obj.id,
            owner_id=request.user,
        ).exists()
