from builtins import object

from django.conf import settings

from rest_framework import serializers

from kelvin.common.staff_reader import staff_reader
from kelvin.resources.serializers import ResourceSerializerCorp

from ..models import User, UserProject
from .base import BaseUserSerializer


class UserSerializer(BaseUserSerializer):
    """
    Сериализатор пользователя.
    """
    avatar = serializers.CharField(source='default_avatar_id')
    roles = serializers.SerializerMethodField()
    settings = serializers.SerializerMethodField()
    projects = serializers.SerializerMethodField()

    def get_roles(self, obj):
        """
        Получаем роли пользователя от микросеврса staff_reader
        """
        user = obj  # self.context['request'].user

        staff_roles = staff_reader.get_user_roles(username=user.username)

        result = {}

        for staff_role_key in list(staff_roles.keys()):
            result['is_' + staff_role_key] = staff_roles[staff_role_key]

        user_boolean_flags = [
            'is_superuser',
            'is_staff',
            'is_teacher',
            'is_student',
            'is_parent',
            'is_content_manager',
            'is_curator',
            'is_tos_accepted',
            'is_support',
        ]

        for flag in user_boolean_flags:
            result[flag] = getattr(user, flag)

        return result

    def get_settings(self, obj):
        """
        Прокидываем настройки для фронта
        """
        return {
            'use_staff_affiliation': (
                settings.YAUTH_ADD_STAFF_AFFILIATION_USER_FIELD
            ),
            'language': staff_reader.get_user_ui_language(obj.username),
        }

    def get_projects(self, obj):
        user_projects = UserProject.objects.filter(user=obj).select_related('project', 'project__nda_resource')
        return UserProjectSerializer(many=True).to_representation(user_projects)

    class Meta(BaseUserSerializer.Meta):
        additional_read_only_fields = (
            'is_superuser',
            'roles',
            'settings',
            'projects',
            'experiment_flags',
        )
        read_only_fields = (
            BaseUserSerializer.Meta.read_only_fields +
            additional_read_only_fields
        )
        fields = (
            BaseUserSerializer.Meta.fields +
            additional_read_only_fields
        )


class SimpleUserSerializer(serializers.ModelSerializer):
    """
    SimpleUserSerializer ...
    """
    class Meta(object):
        model = User
        fields = (
            'first_name',
            'last_name',
            'username',
            'id',
        )


class UserCourseOwnerSerializer(serializers.ModelSerializer):
    class Meta(object):
        model = User
        fields = (
            'id',
            'username',
            'first_name',
            'last_name',
            'middle_name',
        )


class UserProjectSerializer(serializers.ModelSerializer):
    # project_nda_resource = ResourceSerializerCorp()
    slug = serializers.ReadOnlyField(source='project.slug')
    title = serializers.ReadOnlyField(source='project.title')
    nda = serializers.ReadOnlyField(source='project.nda')
    id = serializers.ReadOnlyField(source='project.id')
    add_code = serializers.ReadOnlyField(source='project.add_code')
    nda_resource = ResourceSerializerCorp(source='project.nda_resource')

    class Meta(object):
        model = UserProject
        fields = (
            'project',
            'nda_accepted',
            'slug',
            'id',
            'nda',
            'add_code',
            'title',
            'nda_resource',
        )


class UserCreateSerializer(serializers.ModelSerializer):
    username = serializers.CharField(required=True)
    email = serializers.CharField(required=True)
    avatar = serializers.CharField(source='default_avatar_id', required=False)

    class Meta:
        model = User
        fields = (
            'username',
            'email',
            'first_name',
            'last_name',
            'display_name',
            'avatar',
            'gender',
        )


class UserUpdateSerializer(serializers.ModelSerializer):
    email = serializers.CharField(required=True)
    avatar = serializers.CharField(source='default_avatar_id', required=False)

    class Meta:
        model = User
        fields = (
            'email',
            'first_name',
            'last_name',
            'display_name',
            'avatar',
            'gender',
        )
