from builtins import range
from datetime import timedelta

from celery.utils.log import get_task_logger

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db.models import F
from django.db.models.expressions import Q
from django.utils import timezone

from kelvin.accounts.models import User
from kelvin.accounts.utils import get_user_courses_qs_force, is_user_courses_cache_invalid
from kelvin.celery import app
from kelvin.common.decorators.task_logger import logged_task
from kelvin.common.staff_reader import staff_reader

logger = get_task_logger(__name__)
USER_CHUNK_SIZE = settings.UPDATE_USER_PROFILE_CHUNK_SIZE


@logged_task
@app.task(soft_time_limit=60 * 15, time_limit=60 * 10)
def update_user_profile_chunk(chunk_number, chunk_no):
    logger.info('Processing user profiles chunk: {}'.format(chunk_no))

    users = User.objects.annotate(id_mod_chunk_no=F('id') % chunk_number).filter(id_mod_chunk_no=chunk_no)

    user_info_map = {}  # этот индекс нужен для того, чтобы по ответам стафф-ридера найти объект пользователя
    usernames = []  # список логинов для передачи в стаффридер пачкой

    for user in users:
        username = user.username
        user_info_map[username] = user
        usernames.append(username)

    users_info = staff_reader.get_suggestuser_many(usernames=usernames)

    for info in users_info:
        user = user_info_map[info['login']]
        user.is_dismissed = info.get('official', {}).get('is_dismissed', False)
        user.save()


@logged_task
@app.task()
def update_user_profile():
    # рассчитываем число чанков так, чтобы соблюдать константное число элементов в чанке
    # таким образом мы допускаем, что с ростом числа студентов будет расти число чанков ( тасков в очереди ),
    # но зато какждый из чанков будет обрабатываться +/- константное время

    all_count = User.objects.all().count()

    chunks_count = all_count // USER_CHUNK_SIZE
    if all_count % USER_CHUNK_SIZE > 0:
        chunks_count += 1

    logger.info("Divided {} users into {} chunks (at most {} users per chunk)".format(
        all_count, chunks_count, USER_CHUNK_SIZE
    ))

    for chunk_no in range(chunks_count):
        update_user_profile_chunk.delay(chunks_count, chunk_no)
