from datetime import timedelta

from celery.utils.log import get_task_logger

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.cache import caches
from django.db.models.expressions import Q
from django.utils import timezone

from kelvin.accounts.models import User
from kelvin.accounts.utils import get_user_courses_qs_force, is_user_courses_cache_invalid
from kelvin.celery import app
from kelvin.common.decorators.task_logger import logged_task

main_cache = caches['main_cache']
user_course_cache = caches['user_course']

logger = get_task_logger(__name__)


def set_update_user_courses_task(user_id, use_matchman=False, force_update=False):
    cache_key = settings.USER_CACHE_LAST_THROTTLE_KEY_TMPL.format(user_id)
    if not force_update and user_course_cache.get(cache_key):
        return
    user_course_cache.set(
        cache_key,
        True,
        timeout=settings.USER_COURSE_CACHE_THROTTLE_TIMEOUT,
    )
    update_user_courses_task.delay(user_id, use_matchman, force_update)


@logged_task
@app.task(bind=True, max_retries=10)
def update_user_courses_task(self, user_id, use_matchman=False, force_update=False):
    if force_update or is_user_courses_cache_invalid(user_id):
        user = User.objects.get(id=user_id)
        get_user_courses_qs_force(
            user=user,
            use_matchman=use_matchman,
        )
    cache_key = settings.USER_CACHE_LAST_THROTTLE_KEY_TMPL.format(user_id)
    user_course_cache.delete(cache_key)


def update_last_users_course_cache(last_visit_days_ago, include_nulls=False):
    User = get_user_model()
    conditions = Q(last_login__gt=timezone.now() - timedelta(days=last_visit_days_ago))
    if include_nulls:
        conditions |= Q(last_login__isnull=True)
    users = User.objects.filter(conditions)

    logger.info(f'Update course cache for users: {[user.id for user in users]}')
    for user in users:
        set_update_user_courses_task(user_id=user.id)
