import logging
from builtins import object
from datetime import datetime, timedelta

from django.apps import apps
from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.contenttypes.models import ContentType
from django.contrib.postgres.aggregates.general import ArrayAgg
from django.core.cache import caches
from django.db.models import F, Q
from django.utils import timezone

from kelvin.common.matchman import MatchmanError, matchman
from kelvin.common.staff_reader import StaffReaderError
from kelvin.tags.models import TaggedObject, TagTypeAdapter

User = get_user_model()

logger = logging.getLogger(__name__)
main_cache = caches['main_cache']
user_course_cache = caches['user_course']


def add_project_to_users_cache(user_project):
    if not user_project.project.nda or user_project.project.nda and user_project.nda_accepted:
        cache_key = settings.USER_PROJECTS_CACHE_KEY.format(user_project.user.id)
        cached_user_projects = main_cache.get(cache_key, set([]))
        cached_user_projects.add(user_project.project)
        main_cache.set(cache_key, cached_user_projects)
    else:
        logger.warning("Cannot add nda project to user's projects cache while nda_accepted is false")


def del_project_from_users_cache(user_project):
    cache_key = settings.USER_PROJECTS_CACHE_KEY.format(user_project.user.id)
    cached_user_projects = main_cache.get(cache_key, set([]))
    if user_project.project in cached_user_projects:
        cached_user_projects.remove(user_project.project)
    main_cache.set(cache_key, cached_user_projects)


def get_user_projects(user):
    """ Возвращает все проекты пользователя """
    cache_key = settings.USER_PROJECTS_CACHE_KEY.format(user.id)
    cached_user_projects = main_cache.get(cache_key, set([]))

    if cached_user_projects:
        return cached_user_projects

    user_project_model = apps.get_model('accounts', 'UserProject')
    queryset = (
        user_project_model.objects.filter(user=user, project__nda=True, nda_accepted=True) |
        user_project_model.objects.filter(user=user, project__nda=False)
    )

    cached_user_projects = set([user_project.project for user_project in queryset])
    main_cache.set(cache_key, cached_user_projects)

    return cached_user_projects


def __match_course_by_tags(assignment_rule, user):
    """
    :param course: объект "правило назначения", который содержит формулу матчинга
    :param tags: массив тегов
    :return: True - если набор тегов соответствует формуле, False - в противном случае
    """
    result = False
    for disjunction in assignment_rule.formula:
        disjunction_result = False
        for predicate in disjunction:
            tag_type_class = TagTypeAdapter.get_tag_type_for_semantic_type(predicate['semantic_type'])

            operation_result = tag_type_class.perform_boolean_operation(
                operation=predicate['operation'],
                target_value=predicate['value'],
                object=user,
            )

            if operation_result:
                # если среди предикатов, объединенных через ИЛИ есть хотя бы один истинный,
                # то все объежинение истинно
                disjunction_result = True
                break
        if not disjunction_result:
            # если среди предикатов-дизъюнкций, объединенных через И есть хотя бы один ложный,
            # то все объединение ложно
            result = False
            break

        result = True
    return result


def user_matched_to_course_by_single_rule(assignment_rule, user):
    """ Возвращает True, если пользователь соответствует переданному правилу назначения.
        Валидирует соответствие проекта курса правила назначения и списка проектов, доступных пользователю -
        матчинг не производится, если проект курса правила назначения не в списке проектов пользователя.
    """
    user_projects = get_user_projects(user)
    if assignment_rule.course.project not in user_projects:
        logger.warning("Attempt to match user {} by rule {} from project {} which is not allowed to user".format(
            user.username,
            assignment_rule.id,
            assignment_rule.course.project.id
        ))
        return False

    user.tag_values = get_user_tags(user)
    try:
        return __match_course_by_tags(assignment_rule, user)
    except StaffReaderError as exc:
        logger.warning(repr(exc))


def __get_user_courses_from_cache(user):
    """
    Возвращает закешированные курсы, с которыми матчится пользователь
    Если в кеше для пользователя пусто, то вернет пустой список
    """
    cache_key = settings.USER_CACHE_KEY_TMPL.format(user.id)
    return user_course_cache.get(cache_key, set([]))


def __get_course_users_from_cache(course):
    cache_key = settings.COURSE_CACHE_KEY_TMPL.format(course.id)
    return user_course_cache.get(cache_key, set([]))


def __get_rule_users_from_cache(assignment_rule):
    cache_key = settings.COURSE_RULE_KEY_TMPL.format(assignment_rule.id)
    return user_course_cache.get(cache_key, set([]))


def get_user_rules_from_cache(user):
    cache_key = settings.USER_RULES_CACHE_KEY_TMPL.format(user.id)
    return user_course_cache.get(cache_key, set([]))


def put_rules_to_user_rules_cache(user, assignment_rules, invalidate_user_courses=True):
    user_rules_cache_key = settings.USER_RULES_CACHE_KEY_TMPL.format(user.id)
    cached_rules_ids = user_course_cache.get(user_rules_cache_key, set([]))

    for assignment_rule in assignment_rules:
        if assignment_rule.id not in cached_rules_ids:
            cached_rules_ids.add((assignment_rule.id, assignment_rule.course_id))
            logger.info("Put assignment_rule {} to user's {} matched rules cache".format(assignment_rule.id, user.id))
    # кешируем правила, с которыми матчится пользователь
    user_course_cache.set(
        user_rules_cache_key,
        cached_rules_ids,
        timeout=settings.USER_COURSE_CACHE_TTL
    )
    if invalidate_user_courses:
        # вызываем этот метод для совместимости с предыдущей версией
        invalidate_user_courses_cache(user, drop_reverse=False)


def put_user_to_course_rules_cache(user, assignment_rule):
    course_rule_cache_key = settings.COURSE_RULE_KEY_TMPL.format(assignment_rule.id)
    cached_user_ids = user_course_cache.get(course_rule_cache_key, set([]))

    if user.id not in cached_user_ids:
        cached_user_ids.add(user.id)
        logger.info("Put user {} to course rule's {} matched users cache".format(user.id, assignment_rule.id))
    # кешируем пользователей, сматченных с правилом
    user_course_cache.set(
        course_rule_cache_key,
        cached_user_ids,
        timeout=settings.USER_COURSE_CACHE_TTL
    )

    # вызываем этот метод для совместимости с предыдущей версией
    put_user_to_course_users_cache(user.id, assignment_rule.course_id)


def put_course_to_user_courses_cache(user_id, course_id):
    user_cache_key = settings.USER_CACHE_KEY_TMPL.format(user_id)
    cached_course_ids = user_course_cache.get(user_cache_key, set([]))

    if course_id not in cached_course_ids:
        cached_course_ids.add(course_id)
        logger.info("Put course {} to user's {} matched courses cache".format(course_id, user_id))
        #  кешируем курсы, сматченные с пользователем
        user_course_cache.set(
            user_cache_key,
            cached_course_ids,
            timeout=settings.USER_COURSE_CACHE_TTL
        )


def put_user_to_course_users_cache(user_id, course_id):
    course_cache_key = settings.COURSE_CACHE_KEY_TMPL.format(course_id)
    cached_user_ids = user_course_cache.get(course_cache_key, set([]))

    if user_id not in cached_user_ids:
        cached_user_ids.add(user_id)
        logger.info("Put user {} to course's {} matched users cache".format(user_id, course_id))
    #  кешируем курсы, сматченные с пользователем
    user_course_cache.set(
        course_cache_key,
        cached_user_ids,
        timeout=settings.USER_COURSE_CACHE_TTL
    )


def remove_course_from_user_courses_cache(user_id, course_id):
    """ удаляем курс из кеша курсов пользователя, если он там есть """
    user_cache_key = settings.USER_CACHE_KEY_TMPL.format(user_id)
    cached_course_ids = user_course_cache.get(user_cache_key, set([]))

    if course_id in cached_course_ids:
        cached_course_ids.remove(course_id)
        user_course_cache.set(
            user_cache_key,
            cached_course_ids,
            timeout=settings.USER_COURSE_CACHE_TTL
        )
        logger.info("Remove course {} fropm user's {} matched courses cache".format(course_id, user_id))


def remove_rule_from_user_rules_cache(user_id, assignment_rule_id):
    """ удаляем правило назначения из кеша правил пользователя, если оно там есть """
    user_rule_cache_key = settings.USER_RULES_CACHE_KEY_TMPL.format(user_id)
    cached_rules_ids = user_course_cache.get(user_rule_cache_key, set([]))

    if assignment_rule_id in cached_rules_ids:
        cached_rules_ids.remove(assignment_rule_id)
        user_course_cache.set(
            user_rule_cache_key,
            cached_rules_ids,
            timeout=settings.USER_COURSE_CACHE_TTL
        )
        logger.info("Remove assignment rule {} from user's {} matched rules cache".format(assignment_rule_id, user_id))


def remove_user_from_course_users_cache(user_id, course_id):
    """ удаляем пользователя из кеша пользователей курса, если он там есть """
    course_cache_key = settings.COURSE_CACHE_KEY_TMPL.format(course_id)
    cached_user_ids = user_course_cache.get(course_cache_key, set([]))

    if user_id in cached_user_ids:
        cached_user_ids.remove(user_id)
        user_course_cache.set(
            course_cache_key,
            cached_user_ids,
            timeout=settings.USER_COURSE_CACHE_TTL
        )
        logger.info("Remove user {} from course's {} matched users cache".format(user_id, course_id))


def remove_user_from_rule_users_cache(user_id, assignment_rule_id):
    """ удаляем пользователя из кеша пользователей курса, если он там есть """
    course_rule_cache_key = settings.COURSE_RULE_KEY_TMPL.format(assignment_rule_id)
    cached_user_ids = user_course_cache.get(course_rule_cache_key, set([]))

    if user_id in cached_user_ids:
        cached_user_ids.remove(user_id)
        user_course_cache.set(
            course_rule_cache_key,
            cached_user_ids,
            timeout=settings.USER_COURSE_CACHE_TTL
        )
        logger.info("Remove user {} from course rule's {} matched users cache".format(user_id, assignment_rule_id))


def invalidate_user_courses_cache(user, drop_reverse=True):
    user_cache_key = settings.USER_CACHE_KEY_TMPL.format(user.id)
    user_cache_last_update_key = settings.USER_CACHE_LAST_UPDATE_KEY_TMPL.format(user.id)
    cached_course_ids = user_course_cache.get(user_cache_key, set([]))
    if drop_reverse:
        logger.info("Remove user: {} from reverse cache for courses: {}".format(user.id, cached_course_ids))
        for id in cached_course_ids:
            remove_user_from_course_users_cache(user_id=user.id, course_id=id)
    if cached_course_ids:
        user_course_cache.delete(user_cache_key)
        user_course_cache.delete(user_cache_last_update_key)


def invalidate_course_users_cache(course, drop_reverse=True):
    course_cache_key = settings.COURSE_CACHE_KEY_TMPL.format(course.id)
    cached_user_ids = main_cache.get(course_cache_key, set([]))
    if drop_reverse:
        logger.info("Dropping reverse cache for each course {} users: {}".format(course.id, cached_user_ids))
        for id in cached_user_ids:
            remove_course_from_user_courses_cache(user_id=id, course_id=course.id)
        user_course_cache.delete(course_cache_key)


def invalidate_user_rules_cache(user, drop_reverse=True):
    user_cache_key = settings.USER_RULES_CACHE_KEY_TMPL.format(user.id)
    cached_rules_ids = user_course_cache.get(user_cache_key, set([]))
    if drop_reverse:
        logger.info("Remove user: {} from reverse cache for rules: {}".format(user.id, cached_rules_ids))
        for id in cached_rules_ids:
            remove_user_from_rule_users_cache(user_id=user.id, assignment_rule_id=id)
    user_course_cache.delete(user_cache_key)


def invalidate_rule_users_cache(assignment_rule, drop_reverse=True):
    rule_cache_key = settings.COURSE_RULE_KEY_TMPL.format(assignment_rule.id)
    cached_user_ids = main_cache.get(rule_cache_key, set([]))
    if drop_reverse:
        logger.info("Dropping reverse cache for each rule {} users: {}".format(assignment_rule.id, cached_user_ids))
        for id in cached_user_ids:
            remove_rule_from_user_rules_cache(user_id=id, assignment_rule_id=assignment_rule.id)
        user_course_cache.delete(rule_cache_key)


def user_matched_to_course(user, course, use_matchman=False):
    """
    Возвращает True, если пользователь соответствуюет хотя бы одному правилу назначения курса.
    Возвращает False, если  пользователь не соответствует ни одному правилу назначения курса.
    """

    assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
    if settings.USER_COURSE_CACHE_ENABLE:
        cached_courses = __get_user_courses_from_cache(user)

        if course.id in cached_courses:
            logger.debug("User {} matched to course {} (from cache)".format(user.id, course.id))
            return True

    if use_matchman:
        try:
            return matchman.match_course(user_id=user.id, course_id=course.id)
        except MatchmanError as e:
            logger.error('Matching failed: {}'.format(e))

    matched = False

    assignment_rules_qs = assignment_rule_model.objects.filter(course=course)
    for rule in assignment_rules_qs:
        try:
            user.tag_values = get_user_tags(user)
            if __match_course_by_tags(rule, user):
                # INTLMS-1464 quick fix
                put_rules_to_user_rules_cache(user, [rule])
                put_user_to_course_rules_cache(user, rule)
                put_user_to_course_users_cache(user.id, course.id)
                put_course_to_user_courses_cache(user.id, course.id)
                matched = True
                break
        except Exception as e:
            logger.error("Incorrect rule: {}".format(rule.id))

    # TODO: think about re-build user->courses cache in this function if cache is empty.
    # To do that - we should re-use match-all-courses-for-user logic from function 'get_user_courses_qs'

    return matched or not len(assignment_rules_qs)  # if no rules treat course as matched despite 'mathed' variable


def get_user_courses(user, project):
    """
        Возвращает все курсы, теоретически доступные пользователю в проекте в результате применения правил назначения

        Алгоритм:
        1. получаем все курсы проекта ( порядок "десятки" )
        2. получаем все теги пользователя ( порядок "единицы" )
        3. для каждого курса получаем все его правила назначения ( порядок "единицы" )
        4. для каждого правила назначения провереям, матчится ли оно с набором тегов пользователя
    """
    course_model = apps.get_model('courses', 'Course')

    courses_qs = course_model.objects.filter(project=project)
    if not courses_qs.count():
        """ В переданном проекте нет ни одного курса - возврашать нечего"""
        return []

    assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
    assignment_rules_qs = assignment_rule_model.objects.filter(
        course__project=project
    ).select_related('course')

    matched_courses = []
    user.tag_values = get_user_tags(user)
    for rule in assignment_rules_qs:
        try:
            user.tag_values = get_user_tags(user)
            if __match_course_by_tags(rule, user):
                matched_courses.append(rule.course)
        except Exception as e:
            logger.error("Incorrect rule: {}".format(rule.id))

    return matched_courses


def get_user_courses_qs(user, use_matchman=False, update_cache=False):
    if not settings.USER_COURSE_CACHE_ENABLE and user.is_superuser and not user.is_support:
        user_courses_ids = __get_user_courses_from_cache(user)

        #  если в кеше курсов, сматченных с пользователем что-то есть, то отдаём
        if user_courses_ids:
            logger.info("Getting courses {} for user {} from cache".format(user_courses_ids, user.id))
            return user_courses_ids

    return get_user_courses_qs_force(
        user=user,
        use_matchman=use_matchman,
        update_cache=update_cache,
    )


def get_user_courses_qs_force(user, use_matchman=False, update_cache=True):
    """
        Возвращает все курсы пользователя в проекте в результате применения правил назначения

        Алгоритм:
        1. получаем все курсы проекта ( порядок "десятки" )
        2. получаем все теги пользователя ( порядок "единицы" )
        3. для каждого курса получаем все его правила назначения ( порядок "единицы" )
        4. для каждого правила назначения провереям, матчится ли оно с набором тегов пользователя
    """
    logger.info("Getting courses for user {} from db".format(user.id))
    user_cache_key = settings.USER_CACHE_KEY_TMPL.format(user.id)
    # TODO: refactor code below to use __get __put helpers which are above
    course_model = apps.get_model('courses', 'Course')
    course_student_model = apps.get_model('courses', 'CourseStudent')

    qs = course_model.objects.filter(
        Q(date_closed__isnull=True) | Q(date_closed__gt=timezone.now()),
        free=True,
    ).exclude(
        id__in=course_student_model.objects.filter(
            student_id=user,
            deleted=False,
        ).values_list('course_id', flat=True)
    )
    qs = qs.filter(project__in=get_user_projects(user))

    if user.is_superuser:
        return qs

    if user.is_support:
        return qs.filter(available_for_support=True)

    assignment_rule_model = apps.get_model('courses', 'AssignmentRule')
    assignment_rules_qs = assignment_rule_model.objects.filter(
        course__in=qs
    )

    project_course_ids = list(qs.values_list('id', flat=True))

    if settings.COURSES_NO_RULE_NO_COURSE:
        # строгая логика - если нет правила в курсе, то он недоступен
        matched_courses_ids = set([])
    else:
        # оставляем доступными курсы, у которых нет правил назначения
        matched_courses_ids = set(project_course_ids) - set(assignment_rules_qs.values_list(
            'course_id', flat=True))

    rules_map = assignment_rule_model.objects.in_bulk(assignment_rules_qs)
    t = datetime.now()
    matched_rules_ids = []

    if use_matchman:
        try:
            matched_rules_ids = matchman.get_matched_rules(user.id)
        except MatchmanError as e:
            logger.error('Matching by matchman failed ( falling back to old-style matching ): {}'.format(e))
            use_matchman = False

    if not use_matchman:
        user.tag_values = get_user_tags(user)
        matched_rules_ids = [
            rule.id
            for rule in assignment_rules_qs
            if __match_course_by_tags(rule, user)
        ]
    logger.debug('Match user courses for {} in {}'.format(user.username, datetime.now() - t))

    for rule_id in matched_rules_ids:
        if rule_id not in rules_map:
            continue
        rule = rules_map[rule_id]
        matched_courses_ids.add(rule.course.id)
        if update_cache and settings.USER_COURSE_CACHE_ENABLE:
            course_cache_key = settings.COURSE_CACHE_KEY_TMPL.format(rule.course.id)
            user_course_cached = user_course_cache.get(
                course_cache_key
            )
            #  заполняем вспомогательный кеш из курсов в пользователи,
            #  который нам поможет обнулить пользовательские кеши при изменении конкретного правила назначения
            if user_course_cached:
                user_course_cached.add(user.id)
                user_course_cache.set(
                    course_cache_key,
                    user_course_cached,
                    timeout=settings.USER_COURSE_CACHE_TTL
                )
                logger.info("Updating course {} cache. New users set is: {}".format(
                    rule.course.id, user_course_cached
                ))
            else:
                logger.info("Initializing course {} cache. New users set is: [{}]".format(
                    rule.course.id,
                    user.id,
                ))
                user_course_cache.set(
                    course_cache_key,
                    set([user.id]),
                    timeout=settings.USER_COURSE_CACHE_TTL,
                )

    if update_cache and settings.USER_COURSE_CACHE_ENABLE:
        #  кешируем курсы, сматченные с пользователем
        user_course_cache.set(
            user_cache_key,
            matched_courses_ids,
            timeout=settings.USER_COURSE_CACHE_TTL,
        )
        user_course_cache.set(
            settings.USER_CACHE_LAST_UPDATE_KEY_TMPL.format(user.id),
            timezone.now(),
            timeout=settings.USER_COURSE_CACHE_EXPIRATION_TIMEOUT,
        )

        rules_to_cache = []
        for rule_id in matched_rules_ids:
            if rule_id not in rules_map:
                continue
            rule = rules_map[rule_id]
            rules_to_cache.append(rule)
            put_user_to_course_rules_cache(user, rule)
        put_rules_to_user_rules_cache(user, rules_to_cache, invalidate_user_courses=False)

    return matched_courses_ids


def get_user_courses_from_matchman(user_id, project_id):
    matchman_courses = list(matchman.get_user_courses(user_id, project_id).items())
    return [
        course_id
        for course_id, matched in matchman_courses
        if matched
    ]


def is_user_courses_cache_invalid(user_id):
    cache_key = settings.USER_CACHE_KEY_TMPL.format(user_id)
    if not user_course_cache.get(cache_key):
        return True

    cache_key = settings.USER_CACHE_LAST_UPDATE_KEY_TMPL.format(user_id)
    last_updated = user_course_cache.get(cache_key)
    if not last_updated:
        return True

    return last_updated + timedelta(seconds=settings.USER_COURSE_CACHE_EXPIRATION_TIMEOUT) < timezone.now()


def get_user_tags(user):
    return dict(
        TaggedObject.objects.filter(
            object_id=user.id,
            content_type=ContentType.objects.get_for_model(User),
        ).values('tag__type').annotate(values=ArrayAgg('tag__value')).values_list('tag__type', 'values')
    )


# TODO [REFACTOR] Move to user - в модели User сейчас другая логика
def is_student(user):
    """
    Определение ученика в системе
    """
    return not user.is_authenticated or not (
        user.is_parent or user.is_teacher or user.is_content_manager
    )


class UserCourseFilter(object):
    def __init__(self, courses_query_set, user):
        self.courses_qs = courses_query_set
        self.user = user

    def distinguish_qs(self, qs, fields):
        if not fields:
            return qs.distinct()
        else:
            return qs.order_by(fields).distinct(fields)

    def get_filtered_qs(self):
        raise NotImplementedError("Inherit and redefine 'get_filtered_qs' as you wish")


class UserCourseFilterStudent(UserCourseFilter):
    """ Сужает QuerySet курсов до курсов, где переданный в конструкторе пользователь является студентом """
    def get_filtered_qs(self):
        return self.distinguish_qs(self.courses_qs.filter(students__id=self.user.id), "id")


class UserCourseFilterCurator(UserCourseFilter):
    """ Сужает QuerySet курсов до курсов, где переданный в конструкторе пользователь является куратором"""
    def get_filtered_qs(self):
        return self.distinguish_qs(
            self.courses_qs.annotate(is_curator=F('coursepermission__permission').bitand(
                apps.get_model('courses', 'CoursePermission').CURATOR)
            ).filter(is_curator__gt=0, coursepermission__user=self.user.id).order_by('id').distinct('id'), "id"
        )


class UserCourseFilterStats(UserCourseFilter):
    """Сужает QuerySet курсов до курсов, где переданный в конструкторе пользователь может смотреть статистку"""
    def get_filtered_qs(self):
        necessary_roles_bitmask = 0
        CoursePermission = apps.get_model('courses', 'CoursePermission')
        for role, available_actions in list(CoursePermission.PERMISSION_MATRIX.items()):
            if available_actions[CoursePermission.STATS]:
                necessary_roles_bitmask |= role

        return self.distinguish_qs(
            self.courses_qs.annotate(can_view_stats=F('coursepermission__permission').bitand(
                necessary_roles_bitmask)
            ).filter(can_view_stats__gt=0, coursepermission__user=self.user.id).order_by('id').distinct('id'), "id"
        )


class UserCourseFilterFactory(object):
    AVAILABLE_ROLE_FILTERS = {
        "student": UserCourseFilterStudent,
        "curator": UserCourseFilterCurator,
    }

    AVAILABLE_ACTION_FILTERS = {
        "stats": UserCourseFilterStats,
    }

    @classmethod
    def create_filter_instance_for_role_name(cls, role_name, courses_query_set, user):
        if role_name not in list(cls.AVAILABLE_ROLE_FILTERS.keys()):
            raise RuntimeError("Unsupported user role {}".format(role_name))

        return cls.AVAILABLE_ROLE_FILTERS[role_name](
            courses_query_set,
            user
        )

    @classmethod
    def create_filter_instance_for_action_name(cls, action_name, courses_query_set, user):
        if action_name not in list(cls.AVAILABLE_ACTION_FILTERS.keys()):
            raise RuntimeError("Unsupported user role {}".format(action_name))

        return cls.AVAILABLE_ACTION_FILTERS[action_name](
            courses_query_set,
            user
        )
