import json
import logging
from builtins import str

from django_replicated.decorators import use_master
from rest_condition import And, Or

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.contenttypes.models import ContentType
from django.core.cache import caches
from django.db import IntegrityError, transaction
from django.db.models.expressions import Q
from django.http import Http404

from rest_framework import mixins, status
from rest_framework.decorators import api_view, detail_route, list_route
from rest_framework.exceptions import ValidationError
from rest_framework.generics import CreateAPIView, RetrieveUpdateAPIView
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.settings import api_settings
from rest_framework.viewsets import GenericViewSet

from kelvin.accounts.update_user_courses_task import set_update_user_courses_task
from kelvin.accounts.utils import is_user_courses_cache_invalid
from kelvin.common.permissions import tvm_service_permission_factory
from kelvin.courses.models import Course, CourseStudent
from kelvin.courses.utils import add_courses_to_user
from kelvin.projects.models import Project
from kelvin.tags.models import Tag, TaggedObject, TagTypeStudyGroup

from ..common.pagination import PaginationDefault
from ..tags.tasks_proc.groups import tag_single_user
from .models import StudyGroup, StudyGroupUserLink, UserProject
from .permissions import (
    IsParent, IsStaff, ObjectCanManageStudyGroup, ObjectForContentManager, ObjectIsUserInCourseOwner, ObjectReadOnly,
    ObjectSelf, ObjectStudyGroupOwner,
)
from .serializers import (
    ChildInfoSerializer, ChildSerializer, SimpleUserSerializer, StudyGroupSerializer, UserCreateSerializer,
    UserSerializer, UserUpdateSerializer,
)
from .utils import invalidate_user_courses_cache

logger = logging.getLogger(__name__)
main_cache = caches['main_cache']
User = get_user_model()


IsFrontendServicePermission = tvm_service_permission_factory([settings.TVM_FRONTEND])


@transaction.atomic
def create_profile(request):
    return Response(status=status.HTTP_410_GONE)
    # """
    # Create profile and add staff affiliation to user object, when it creates
    # on first login.
    # Method copied and customized from `django_yauth.views.create_profile`
    # """
    # if not request.user.is_authenticated():
    #     return HttpResponseForbidden('Authentication required')
    #
    # if get_setting(['CREATE_PROFILE_ON_ACCESS', 'YAUTH_CREATE_PROFILE_ON_ACCESS']):
    #     for model in get_profile_models():
    #         create_profile(request.yauser, model)
    #
    # if get_setting(['CREATE_USER_ON_ACCESS', 'YAUTH_CREATE_USER_ON_ACCESS']):
    #     created_user = create_user(request.yauser)
    #     add_user_to_default_project(created_user)
    #
    # redirect_to = request.POST.get('next', request.GET.get('next', ''))
    #
    # if not is_safe_url(redirect_to, request.get_host()):
    #     return HttpResponseForbidden('Forbidden redirect location')
    #
    # return HttpResponseRedirect(redirect_to)


def add_user_to_default_project(user):
    default_project = get_default_project()
    if default_project:
        add_user_to_project(user, default_project)
        tag_user(user, default_project)


def get_default_project():
    return Project.objects.filter(pk=1, slug='DEFAULT').first()


def add_user_to_project(user, project):
    UserProject.objects.get_or_create(
        user=user,
        project=project,
    )


def tag_user(user, project):
    tagged_objects, _ = tag_single_user(user=user, project_id=project.id)
    TaggedObject.objects.bulk_create(tagged_objects)


@use_master
@api_view(http_method_names=['POST'])
def auto_add_to_project(request, **kwargs):
    """ Simple API view to self-add user to project """

    # TODO: провалидировать add_code (string) и nda_accepted (boolean)
    data = json.loads(request.body or '[]')

    for project_joining in data:
        add_code = project_joining.get('add_code')
        nda_accepted = project_joining.get('nda_accepted')

        if not add_code:
            return Response(
                data='Project code is empty',
                status=status.HTTP_400_BAD_REQUEST
            )

        try:
            project = Project.objects.get(add_code=add_code)
        except Project.DoesNotExist:
            return Response(
                data='Cannot find a project by this code: {}' . format(add_code),
                status=status.HTTP_404_NOT_FOUND
            )

        # Если у нас глобально отключено автосоздание пользователя, то по ссылке на самозапиливание в проект
        # пользователя надо заводить обязательно
        # if not get_feature_bool_option('YAUTH_CREATE_USER_ON_ACCESS', True):
        #     moebius_user = create_user(request.yauser)
        #     logger.debug("Created user {} by direct project link".format(moebius_user.username))
        # else:
        moebius_user = request.user

        try:
            cache_key = settings.USER_PROJECTS_CACHE_KEY.format(moebius_user.id)
            cached_user_projects = main_cache.get(cache_key, set([]))

            UserProject.objects.update_or_create(
                user=moebius_user,
                project=project,
                defaults=dict(
                    nda_accepted=bool(nda_accepted),
                ),
            )

            cached_user_projects.add(project)
            main_cache.set(cache_key, cached_user_projects)

        except Exception as e:
            return Response(
                data='Error while selfadding to a project: {}'.format(str(e)),
                status=status.HTTP_400_BAD_REQUEST
            )

    # Надо сбросить кеш курсов пользователя, для того, чтобы при заходе в библиотеку
    # ему перематчились курсы с учетом добавления в проект
    invalidate_user_courses_cache(moebius_user, drop_reverse=True)

    return Response(
        status=status.HTTP_201_CREATED,
        data={'project_id': project.id},
        content_type='application/json',
    )


@api_view(http_method_names=['POST'])
def create_user_for_project(request, project_id, **kwargs):
    """
        Создаем пользователя и добавляем его в проект.
        Сделано для интеграции с "Лицеем"
        FIXME: убрать в отдельное kelvin-приложение lyceum в модуль views
    """
    if not project_id:
        msg = 'project_id is empty'
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_400_BAD_REQUEST
        )

    try:
        project_id = int(project_id)
    except ValueError:
        msg = 'project_id must be an integer'
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_400_BAD_REQUEST
        )

    project = None

    try:
        project = Project.objects.get(id=project_id)
    except Project.DoesNotExist:
        msg = 'Cannot find a project by its id: {}' . format(project_id)
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_404_NOT_FOUND
        )

    username = request.data.get('username', None)
    if not username:
        msg = 'username is required and should be non empty'
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_400_BAD_REQUEST
        )

    yauid = request.data.get('yauid', None)
    if not yauid:
        msg = 'yauid is required and should be non empty'
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_400_BAD_REQUEST
        )

    email = request.data.get('email', '')
    first_name = request.data.get('first_name', '')
    last_name = request.data.get('last_name', '')

    courses_ids = request.data.get('courses_ids', None)
    if not courses_ids or not isinstance(courses_ids, list):
        msg = 'courses is required and should be non empty list of courses to be added to',
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_400_BAD_REQUEST
        )

    user, created = User.objects.update_or_create(
        username=username,
        defaults={
            'yauid': yauid,
            'email': email,
            'first_name': first_name,
            'last_name': last_name,
        }
    )

    if not created:
        logger.warning('User "{}" already exists. No need to create it.'.format(username))
    else:
        logger.info('Created new user {}'.format(username))

    try:
        UserProject.objects.get_or_create(
            user=user,
            project=project
        )
    except Exception as e:
        msg = 'Error while adding user: "{}" to a project: "{}". {}'.format(username, project.id, str(e))
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_400_BAD_REQUEST
        )

    courses = Course.objects.filter(project_id=project_id, id__in=courses_ids)

    if courses.count() != len(courses_ids):
        msg = 'At least one of the courses {} are absent in project {}'.format(
            str(courses_ids),
            project_id
        )
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_400_BAD_REQUEST
        )

    try:
        add_courses_to_user(courses, user)
    except Exception as e:
        msg = 'Something wrong duting assigning courses to user: {}. {}'.format(username, str(e))
        logger.error(msg)
        return Response(
            data=msg,
            status=status.HTTP_400_BAD_REQUEST
        )

    return Response(status=status.HTTP_201_CREATED)


class BaseUserViewSet(
    mixins.CreateModelMixin,
    mixins.RetrieveModelMixin,
    mixins.UpdateModelMixin,
    mixins.DestroyModelMixin,
    GenericViewSet,
):
    """
    Рест-ручки пользователя, регистрация пользователя

    Отсутствует ручка списка пользователей
    """
    queryset = User.objects.select_related('teacher_profile')
    permission_classes = (
        IsAuthenticated,
        Or(
            ObjectSelf,
            And(
                ObjectReadOnly,
                Or(
                    ObjectForContentManager,
                    ObjectIsUserInCourseOwner,
                ),
            ),
        ),
    )

    # def get_serializer_context(self):
    #     """
    #     Добавляет в контекcт яндексового пользователя
    #     """
    #     context = super(BaseUserViewSet, self).get_serializer_context()
    #     context['yauser'] = self.request._request.yauser.fields
    #     return context

    @list_route(methods=['POST'], permission_classes=[IsAuthenticated])
    def accept_tos(self, request):
        """
        Проставить текущему пользователю согласие
        с правилами пользования сервисом
        """
        user = request.user

        if user.is_tos_accepted:
            raise ValidationError('TOS is already accepted')

        user.is_tos_accepted = True
        user.save(update_fields=['is_tos_accepted'])

        return Response()

    @list_route()
    def me(self, request, **kwargs):
        """
        Вернуть профиль авторизованного пользователя
        """
        if settings.USER_COURSE_CACHE_ENABLE and is_user_courses_cache_invalid(request.user.id):
            set_update_user_courses_task(request.user.id)

        data = self.get_serializer().to_representation(request.user)

        if not request.user.is_teacher:
            # для не учителей скрываем поле
            data.pop('teacher_profile')

        if request.user.is_parent:
            children = request.user.parent_profile.children.all().only(
                'id',
                'first_name',
                'last_name',
                'avatar',
                'username',
            )

            # курсы детей
            course_students = (
                CourseStudent.objects.filter(student__in=children)
                .order_by('course')
                .select_related('course')
            )
            student_courses = {}
            for course_student in course_students:
                student_courses.setdefault(course_student.student_id, []).append(
                    (course_student.course.id, course_student.course.name)
                )

            data['children'] = (
                ChildSerializer(
                    many=True,
                    context={'courses': student_courses},
                ).to_representation(children)
            )

        return Response(data)

    @list_route(methods=['POST'], permission_classes=[IsAuthenticated, IsParent])
    def add_child(self, request):
        """
        Добавить ребенка по его коду
        """
        if 'code' in request.data:
            try:
                child = User.objects.get(parent_code=request.data['code'])
            except (ValueError, TypeError):
                raise ValidationError('wrong code value')
            except User.DoesNotExist:
                return Response(status=status.HTTP_404_NOT_FOUND)

            request.user.parent_profile.children.add(child)
            return self.me(request)
        raise ValidationError('code was expected')

    @list_route(permission_classes=[IsStaff])  # FIXME temporary permission
    def child(self, request):
        try:
            child = User.objects.get(parent_code=request.query_params['code'])
        except KeyError:
            raise ValidationError('code parameter is required')
        except User.DoesNotExist:
            return Response(status=status.HTTP_404_NOT_FOUND)

        return Response(ChildInfoSerializer().to_representation(child))

    @list_route(methods=['POST'], permission_classes=[IsAuthenticated])
    def add_viewed_hint(self, request):
        """
        Create/change hint
        :param request:
            POST:
                hint_id - required - id of hint
                hint_data - (not required, default {}) - data of hint
        :return:
            200 - hint was changed
            201 - hint was added
        :exception:
            hint_id was not passed
        """

        if 'hint_id' not in request.data:
            raise ValidationError(u"'hint_id' is required")
        hint_id = request.data['hint_id']
        hint_data = request.data.get('hint_data', {})
        user = request.user

        status = 200 if hint_id in user.viewed_hints else 201
        user.viewed_hints[hint_id] = hint_data
        user.save()

        return Response(user.viewed_hints, status=status)


class UserViewSet(BaseUserViewSet):
    serializer_class = UserSerializer
    pagination_class = PaginationDefault

    @staticmethod
    def __validate_suggest_params(request):
        q = request.query_params.get('q', None)
        if not q:
            raise ValidationError("'q' parameter is mandatory")

    # def get_suggest_queryset_trgm(self, query_string):
    #     return list(User.objects.raw("""SELECT id, first_name, last_name, username FROM accounts_user
    #     WHERE (first_name || ' ' || last_name || ' ' || username) %% ( %s )
    #     ORDER BY lower(last_name), lower(first_name), username""", (query_string, )))

    def get_suggest_queryset(self, query_string):
        return User.objects.filter(
            Q(username__icontains=query_string) |
            Q(first_name__icontains=query_string) |
            Q(last_name__icontains=query_string)
        ).order_by('last_name')

    def get_suggest_serializer(self, *args, **kwargs):
        serializer_class = SimpleUserSerializer
        kwargs['context'] = self.get_serializer_context()
        return serializer_class(*args, **kwargs)

    @list_route(methods=['get'])
    def suggest(self, request):
        UserViewSet.__validate_suggest_params(self.request)
        query_string = request.query_params.get('q', None)
        queryset = self.get_suggest_queryset(query_string)
        serializer = self.get_suggest_serializer(queryset[:api_settings.PAGE_SIZE], many=True)
        return Response(serializer.data)


class SiriusUserViewSet(UserViewSet):
    """
    API пользователя. Перегружен сериализатор.
    """
    serializer_class = UserSerializer
    lookup_value_regex = '\d+'

    @list_route(permission_classes=[IsAuthenticated])
    def info(self, request):
        usernames = request.GET.get('usernames', '')
        if not usernames:
            raise ValidationError('usernames is required')
        usernames = usernames.split(',')

        users = User.objects.filter(username__in=usernames)
        return Response(
            SimpleUserSerializer(many=True).to_representation(users)
        )


class StudyGroupViewSet(
    mixins.ListModelMixin,
    mixins.CreateModelMixin,
    mixins.RetrieveModelMixin,
    mixins.UpdateModelMixin,
    GenericViewSet,
):
    serializer_class = StudyGroupSerializer
    pagination_class = None
    permission_classes = [
        IsAuthenticated,
        Or(
            ObjectStudyGroupOwner,
            ObjectCanManageStudyGroup,
        )
    ]
    lookup_value_regex = '\d+'

    def get_queryset(self):
        queryset = StudyGroup.objects.filter().prefetch_related('students')

        if not self.request.user.has_perm('accounts.can_manage_groups'):
            queryset = queryset.filter(owner=self.request.user)

        return queryset

    @staticmethod
    def get_students(request, study_group):
        """Поулчить список студентов учебной группы."""
        return Response(
            SimpleUserSerializer(many=True).to_representation(study_group.students.all())
        )

    @staticmethod
    def add_students(request, study_group):
        """Добавить студентов в учебную группу."""
        usernames = request.data.get('usernames', [])
        existing_students_usernames = study_group.students.values_list('username', flat=True)
        new_students = User.objects.filter(username__in=set(usernames) - set(existing_students_usernames))
        new_student_links = StudyGroupUserLink.objects.bulk_create(
            [StudyGroupUserLink(user=student, study_group=study_group) for student in new_students]
        )
        tag = Tag.objects.filter(
            project=study_group.project,
            type=TagTypeStudyGroup.get_db_type(),
            value=study_group.slug,
        ).first()
        if tag:
            user_content_type = ContentType.objects.get_for_model(User)
            TaggedObject.objects.bulk_create(
                [
                    TaggedObject(tag=tag, object_id=link.user_id, content_type=user_content_type)
                    for link in new_student_links
                ]
            )
        else:
            logger.error('Can\'t find StudyGroup Tag {}'.format(study_group.slug))

        return Response(
            SimpleUserSerializer(many=True).to_representation(new_students)
        )

    @staticmethod
    def remove_students(request, study_group):
        """Удалить студентов из учебной группы."""
        usernames = request.data.get('usernames', [])
        tag = Tag.objects.filter(
            project=study_group.project,
            type=TagTypeStudyGroup.get_db_type(),
            value=study_group.slug,
        ).first()
        if tag:
            user_content_type = ContentType.objects.get_for_model(User)
            user_ids = User.objects.filter(username__in=usernames).values_list('id', flat=True)
            TaggedObject.objects.filter(tag=tag, object_id__in=user_ids, content_type=user_content_type).delete()
        else:
            logger.error('Can\'t find StudyGroup Tag {}'.format(study_group.slug))
        deleted = StudyGroupUserLink.objects.filter(
            study_group=study_group,
            user__username__in=usernames,
        ).delete()

        return Response({"deleted": deleted})

    @detail_route(methods=['get', 'post', 'delete'])
    def students(self, request, pk):
        """Набор ручек для работы с учениками учебной группы."""

        study_group = self.get_object()  # permissions are checked here
        methods_mapping = {
            'GET': self.get_students,
            'POST': self.add_students,
            'DELETE': self.remove_students,
        }

        return methods_mapping[request.method](request, study_group)


class UserProfileView(RetrieveUpdateAPIView, CreateAPIView):
    serializer_class = UserSerializer
    pagination_class = None
    permission_classes = [
        IsFrontendServicePermission,
    ]

    def get_object(self):
        tvm_uid = getattr(self.request, 'tvm_uid', None)
        user = self.request.user
        if not user.is_authenticated:
            # возвращаем 404, если профиль не существует
            if tvm_uid is not None:
                user_exists = User.objects.filter(yauid=tvm_uid).exists()
                if not user_exists:
                    raise Http404

            return self.permission_denied(self.request, 'Not Authenticated')

        if settings.USER_COURSE_CACHE_ENABLE and is_user_courses_cache_invalid(self.request.user.id):
            set_update_user_courses_task(self.request.user.id)

        return self.request.user

    def retrieve(self, request, *args, **kwargs):
        instance = self.get_object()
        data = self.get_serializer().to_representation(instance)

        if not request.user.is_teacher:
            data.pop('teacher_profile')

        return Response(data)

    def update(self, request, *args, **kwargs):
        partial = kwargs.pop('partial', False)
        instance = self.get_object()
        input_serializer = UserUpdateSerializer(instance, data=request.data, partial=partial)
        input_serializer.is_valid(raise_exception=True)
        instance = input_serializer.save()

        data = self.get_serializer().to_representation(instance)
        return Response(data)

    def create(self, request, *args, **kwargs):
        tvm_uid = getattr(request, 'tvm_uid', None)
        if not tvm_uid:
            self.permission_denied(request, 'No user ticket provided.')

        input_serializer = UserCreateSerializer(data=request.data)
        input_serializer.is_valid(raise_exception=True)

        user_data = input_serializer.validated_data

        created = False
        try:
            user, created = User.objects.get_or_create(
                yauid=tvm_uid,
                defaults=user_data
            )
        except IntegrityError as exc:
            from psycopg2 import errorcodes

            pgcode = exc.__cause__.pgcode
            constraint_name = exc.__cause__.diag.constraint_name
            constraint_error_names = {'accounts_user_username_key', 'accounts_user_email_key'}

            if pgcode == errorcodes.UNIQUE_VIOLATION and constraint_name in constraint_error_names:
                username = user_data.pop('username')
                email = user_data.pop('email')
                logger.info("Updating user %s with yauid", username)

                user = User.objects.get(username=username)
                user.yauid = tvm_uid

                if not user.email:
                    user.email = email

                for key, value in user_data.items():
                    setattr(user, key, value)

                user.save()
            else:
                raise exc

        if created:
            user.set_unusable_password()
            user.save()

        add_user_to_default_project(user)

        data = self.get_serializer().to_representation(user)
        return Response(data, status=status.HTTP_201_CREATED)
