import logging
from typing import Tuple

from requests import Session
from requests.exceptions import HTTPError

from django.conf import settings

logger = logging.getLogger(__name__)


def get_level(value: str) -> Tuple[int, bool]:
   incr = value[0] in ("+", "-")
   return int(value), incr


class AchieveryHTTPError(HTTPError):
    pass


class AchieveryClient(object):
    def __init__(self, api_url, token, tester_login):
        self.API_URL = api_url
        self.TOKEN = token
        self.TESTER_LOGIN = tester_login
        self.HEADERS = {
            'Authorization': 'OAuth {token}'.format(token=self.TOKEN),
            'Content-Type': 'application/json',
        }

    @staticmethod
    def get_session():
        return Session()

    def make_request(self, method, url, params=None, session=None, **kwargs):
        session = session or self.get_session()
        kwargs['headers'] = self.HEADERS
        if params:
            kwargs['params'] = params
        resp = session.request(method, url, allow_redirects=False, **kwargs)
        if resp.status_code < 300:
            return resp.json()

        logger.exception(
            'Achievery API error: status_code={}, reason={}, text={}'.format(
                resp.status_code, resp.reason, resp.text
            )
        )

        raise AchieveryHTTPError(response=resp)

    def get_achievement(self, achievment_id, fields=None, session=None):
        url = f'{self.API_URL}/achievements/{achievment_id}/'
        params = {'_fields': ','.join(fields)} if fields else None

        return self.make_request(method='GET', url=url, params=params, session=session)

    def check_ever_given(self, achievment_id, login, session=None):
        url = f'{self.API_URL}/given/'
        params = {'achievement.id': achievment_id, 'person.login': login, '_fields': 'id,revision,level,is_active'}

        resp = self.make_request(method='GET', url=url, params=params, session=session)

        return {
            'id': resp['objects'][0]['id'],
            'revision': resp['objects'][0]['revision'],
            'level': resp['objects'][0]['level'],
            'is_active': resp['objects'][0]['is_active'],
        } if resp['objects'] else None

    def _post_achievement(self, achievment_id, login, level, comment, is_active, fields=None, session=None):
        url = f'{self.API_URL}/given/'
        params = {'_fields': ','.join(fields)} if fields else None

        data = {
            'person.login': login,
            'achievement.id': achievment_id,
            'comment': comment,
            'is_active': is_active,
            'level': level,
        }

        return self.make_request(method='POST', url=url, params=params, json=data, session=session)

    def _put_achievement(self, given_id, revision, level=None, comment=None, is_active=None, fields=None, session=None):
        url = f'{self.API_URL}/given/{given_id}/'
        params = {'_fields': ','.join(fields)} if fields else None
        data = {
            'revision': revision,
        }
        if level is not None:
            data['level'] = level
        if comment is not None:
            data['comment'] = comment
        if is_active is not None:
            data['is_active'] = is_active

        return self.make_request(method='PUT', url=url, params=params, json=data, session=session)

    def change_achievement(
        self,
        achievement_id,
        login,
        comment,
        is_active,
        level,
        fields=None,
        can_decrease_level=False,
        request_if_exists=False,
        session=None
    ):
        session = session or self.get_session()
        kwargs = {
            'comment': comment,
            'fields': fields,
            'is_active': is_active,
            'session': session,
        }
        given_data = self.check_ever_given(achievment_id=achievement_id, login=login, session=session)
        level, incr = get_level(level)
        if given_data:
            # если ачивка неактивна и остается неактивна, то ничего не делаем
            if not given_data['is_active'] and not is_active:
                return

            old_level = given_data['level']
            if incr:  # изменение уровня
                new_level = max(1, old_level + int(level))  # нельзя опускаться ниже 1-го уровня
            else:  # точный уровень
                level = -1 if level == 0 else level  # в API ачивницы (-1) - ачивка без уровней

                # можно ли уменьшать уровень?
                if not can_decrease_level and old_level >= level:
                    new_level = old_level
                else:
                    new_level = level

                # можно ли перевыдавать ачивку, если уровень не поменялся?
                if not request_if_exists and given_data['is_active'] == is_active and old_level == new_level:
                    return None

            old_revision = given_data['revision']

            # если ачивка неактивна, то сначала ее надо активировать отдельным запросом
            if not given_data['is_active'] and is_active:
                return self._put_achievement(
                    session=session,
                    given_id=given_data['id'],
                    level=given_data['level'],
                    revision=old_revision,
                    is_active=True,
                )
                old_revision += 1

            kwargs.update(
                {
                    'given_id': given_data['id'],
                    'revision': old_revision,
                    'level': new_level,
                },
            )

            return self._put_achievement(**kwargs)
        else:
            if level < 0:  # нельзя опускаться ниже 1-го уровня
                level = 1
            level = -1 if level == 0 else level  # в API ачивницы (-1) - ачивка без уровней
            kwargs.update(
                {
                    'achievment_id': achievement_id,
                    'login': login,
                    'level': level,
                },
            )

            return self._post_achievement(**kwargs)

    def check_achievement(self, achievement_id, level, fields=None, session=None):
        session = session or self.get_session()
        if fields and 'revision' not in fields:
            fields.append('revision')
        if fields and 'level' not in fields:
            fields.append('level')

        return self.change_achievement(
            achievement_id,
            login=self.TESTER_LOGIN,
            level=level,
            comment='test give achievement',
            is_active=True,
            fields=fields,
            request_if_exists=True,
            session=session,
        )


client = AchieveryClient(
    api_url=settings.ACHIEVERY_API_URL,
    token=settings.ACHIEVERY_TOKEN,
    tester_login=settings.ACHIEVERY_TESTER_LOGIN,
)
