from django_object_actions import DjangoObjectActions, takes_instance_or_queryset

from django.contrib import admin, messages
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import build_dropdown_filter_with_title

from .achievery_client import AchieveryHTTPError
from .achievery_client import client as achievery_client
from .models import AchievementTeam, UserAchievement
from .services.grant_achievement import grant_achievement
from .services.update_achievement_team_users import update_achievement_team_users


@admin.register(UserAchievement)
class UserAchievementAdmin(DjangoObjectActions, admin.ModelAdmin):
    readonly_fields = (
        'created',
        'modified',
    )
    list_display = (
        'user',
        'achievement_id',
        'level',
        'published_at',
        'criterion',
        'comment',
        'is_active',
    ) + readonly_fields
    raw_id_fields = (
        'user',
        'criterion',
    )
    list_filter = (
        'achievement_id',
        'is_active',
        ('criterion__clesson__course', build_dropdown_filter_with_title('Курс')),
    )
    list_select_related = raw_id_fields
    search_fields = (
        'user__username',
    )

    change_actions = (
        'check_achievement',
        'grant_achievement_force',
    )

    actions = (
        'grant_achievement_force',
    )

    def check_achievement(self, request, obj):
        try:
            achievery_client.check_achievement(
                achievement_id=obj.achievement_id,
                level=obj.level,
            )
            message = _('Ачивка {} готова к использованию').format(obj.achievement_id)
            message_level = messages.SUCCESS

        except AchieveryHTTPError as ex:
            message = _('Ошибка выдачи ачивки: status code: {}, reason: {}, text: {}').format(
                ex.response.status_code,
                ex.response.reason,
                ex.response.text,
            )
            message_level = messages.ERROR

        self.message_user(request=request, message=message, level=message_level)

    check_achievement.label = _("Проверить ачивку")
    check_achievement.short_description = _("Проверить ачивку")

    @takes_instance_or_queryset
    def grant_achievement_force(self, request, queryset):
        for obj in queryset:
            try:
                grant_achievement(user_achievement=obj, force=True)
                message = _('Ачивка выдана')
                message_level = messages.SUCCESS
            except AchieveryHTTPError as ex:
                message = _('Ошибка выдачи ачивки: status code: {}, reason: {}, text: {}').format(
                    ex.response.status_code,
                    ex.response.reason,
                    ex.response.text,
                )
                message_level = messages.ERROR

            self.message_user(request=request, message=message, level=message_level)

    grant_achievement_force.label = _("Выдать ачивку")
    grant_achievement_force.short_description = _("Выдать ачивку")


class UserAchievementInlineAdmin(admin.TabularInline):
    model = UserAchievement

    readonly_fields = (
        'user',
        'achievement_id',
        'level',
        'published_at',
        'criterion',
        'comment',
        'is_active',
        'created',
        'modified',
    )
    fields = readonly_fields
    extra = 0

    can_delete = False

    def has_add_permission(self, request):
        return False


@admin.register(AchievementTeam)
class AchievementTeamAdmin(DjangoObjectActions, admin.ModelAdmin):
    list_display = ('__str__', 'captain_login', 'status', 'clesson_id')
    raw_id_fields = (
        'captain',
        'participants',
    )
    list_select_related = ('captain',)
    list_filter = (
        'clesson_id',
        'status',
    )

    search_fields = (
        'captain_login',
        'participants_logins',
    )

    actions = [
        'update_achievement_team_users_action',
    ]

    change_actions = [
        'update_achievement_team_users_action',
    ]

    @takes_instance_or_queryset
    def update_achievement_team_users_action(self, request, queryset):
        for achievement_team in queryset:
            update_achievement_team_users(achievement_team=achievement_team, force=True)
            level = messages.INFO
            message = _(f"{achievement_team}: участники обновлены")
            if achievement_team.status == AchievementTeam.STATUS_ERROR:
                level = messages.ERROR
                message = _(f"{achievement_team}: ошибка обновления участников")

            self.message_user(
                request=request,
                message=message,
                level=level,
            )

    update_achievement_team_users_action.short_description = _("обновить участников")
    update_achievement_team_users_action.label = _("обновить участников")
