from model_utils.models import TimeStampedModel

from django.contrib.auth import get_user_model
from django.contrib.auth.validators import UnicodeUsernameValidator
from django.contrib.postgres.fields import JSONField
from django.core.validators import RegexValidator
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.validators import list_of_logins_validator

User = get_user_model()
ACHIEVEMENT_LEVEL_REGEXP = r"^[+-]?\d+$"
achievement_level_validator = RegexValidator(ACHIEVEMENT_LEVEL_REGEXP)


@python_2_unicode_compatible
class UserAchievement(TimeStampedModel, models.Model):
    user = models.ForeignKey(to='accounts.User', verbose_name=_('пользователь'))
    achievement_id = models.IntegerField(verbose_name=_('ID ачивки'))
    level = models.CharField(
        verbose_name=_('уровень'),
        max_length=10,
        validators=[achievement_level_validator],
        help_text=_("n - точный уровень (0 - без уровня), +/-n - изменение уровня на n"),
    )
    published_at = models.DateTimeField(null=True, blank=True, verbose_name=_('дата публикации'))
    criterion = models.ForeignKey(to='courses.Criterion', null=True, blank=True, verbose_name=_('критерий'))
    comment = models.CharField(blank=True, max_length=255, verbose_name=_('комментарий'))
    is_active = models.BooleanField(default=False, verbose_name=_('выдана'))
    request_if_exists = models.BooleanField(
        default=False, verbose_name=_('перезапросить ачивку'), help_text=_('перезапросить ачивку, если она уже выдана'),
    )

    def __str__(self):
        return '{} ({}) - {}'.format(
            self.user, self.achievement_id, _('выдана') if self.is_active else _('отобрана'),
        )

    class Meta:
        verbose_name = _('выданная ачивка')
        verbose_name_plural = _('выданные ачивки')


class AchievementTeam(TimeStampedModel, models.Model):
    STATUS_PENDING = 'pending'
    STATUS_READY = 'ready'
    STATUS_ERROR = 'error'
    STATUS_CHOICES = (
        (STATUS_PENDING, _("в обработке")),
        (STATUS_READY, _("готова")),
        (STATUS_ERROR, _("ошибка")),
    )

    name = models.CharField(
        _("Название"),
        max_length=255,
    )
    captain_login = models.CharField(
        _("Логин капитана"),
        max_length=255,
        validators=[UnicodeUsernameValidator],
    )
    participants_logins = JSONField(
        verbose_name=_("логины участников"),
        default=list,
        blank=True,
        validators=[list_of_logins_validator],
    )
    captain = models.ForeignKey(
        to=User,
        verbose_name=_("Капитан"),
        related_name='captain_in_achievement_teams',
        null=True,
        blank=True,
    )
    participants = models.ManyToManyField(
        to=User,
        verbose_name=_("Участники"),
        related_name='participant_in_achievement_teams',
        blank=True,
    )
    status = models.CharField(
        _("статус"),
        max_length=16,
        choices=STATUS_CHOICES,
        default=STATUS_PENDING,
    )
    error_messages = models.TextField(
        _("сообщение об ошибке"),
        blank=True,
    )
    clesson_id = models.PositiveIntegerField(
        verbose_name=_("ID модуля"),
    )

    class Meta:
        verbose_name = _('команда для ачивок')
        verbose_name_plural = _('команды для ачивок')

    def __str__(self):
        return self.name if self.name else f"Team of {self.captain_login}"
