import logging

from django.contrib.auth import get_user_model
from django.db import transaction
from django.utils.translation import ugettext_lazy as _

from kelvin.common.staff_reader import staff_reader

from ..models import AchievementTeam

BULK_BATCH_SIZE = 1000
User = get_user_model()
logger = logging.getLogger(__name__)


def update_achievement_team_users(achievement_team: AchievementTeam, force: bool = False):
    if achievement_team.status != AchievementTeam.STATUS_PENDING and not force:
        return

    with transaction.atomic():
        logins_to_find = set(achievement_team.participants_logins or {})
        logins_to_find.add(achievement_team.captain_login)

        not_found_users_logins = (
            logins_to_find -
            set(User.objects.filter(username__in=logins_to_find).values_list('username', flat=True))
        )

        users_info = staff_reader.get_suggestuser_many(usernames=list(not_found_users_logins))

        User.objects.bulk_create(
            objs=[
                User(
                    username=user_info['login'],
                    email=user_info['work_email'],
                    first_name=user_info['name']['first']['ru'],
                    last_name=user_info['name']['last']['ru'],
                    yauid=int(user_info['uid']),
                    is_dismissed=user_info['official']['is_dismissed'],
                )
                for user_info in users_info
            ],
            batch_size=BULK_BATCH_SIZE,
        )

        captain_to_add = User.objects.filter(username=achievement_team.captain_login).first()

        participants_to_add = User.objects.filter(username__in=achievement_team.participants_logins)
        participants_to_add_count = participants_to_add.count()

        if participants_to_add_count == len(achievement_team.participants_logins) and captain_to_add:
            achievement_team.participants.set(participants_to_add)

            achievement_team.captain = captain_to_add
            achievement_team.status = AchievementTeam.STATUS_READY

        else:
            logger.error(
                _("При обработке команды для ачивки %s не найдены логины %s") % (
                    achievement_team.id,
                    ', '.join(not_found_users_logins),
                ),
            )
            achievement_team.status = AchievementTeam.STATUS_ERROR

        achievement_team.save()
