import traceback

from celery.utils.log import get_task_logger

from django.conf import settings
from django.db import transaction
from django.utils.translation import ugettext_lazy as _

from kelvin.celery import app
from kelvin.common.decorators.task_logger import logged_task
from kelvin.results.models import CourseLessonResult
from kelvin.results.services.pull_triggers import pull_triggers

from .achievery_client import AchieveryHTTPError
from .models import AchievementTeam, UserAchievement
from .services.grant_achievement import grant_achievement
from .services.update_achievement_team_users import update_achievement_team_users

logger = get_task_logger(__name__)


@logged_task
@app.task(bind=True, max_retries=10)
def grant_achievement_task(self, user_achievement_id):
    user_achievement = UserAchievement.objects.filter(id=user_achievement_id).first()
    if user_achievement is not None:
        try:
            grant_achievement(user_achievement=user_achievement)
        except AchieveryHTTPError as exc:
            raise self.retry(exc=exc, countdown=settings.CELERY_DEFAULT_COUNTDOWN)
    else:
        logger.error('UserAchievement with id {} not found'.format(user_achievement_id))


@logged_task
@app.task(max_retries=10)
@transaction.atomic()
def update_achievement_team_users_task(achievement_team_id: int, force: bool = False):
    achievement_team = None
    try:
        achievement_team = (
            AchievementTeam.objects
            .filter(id=achievement_team_id)
            .select_for_update(skip_locked=True)
            .first()
        )

        if achievement_team is None:
            logger.error(_("AchievementTeam %s locked or does not exist") % achievement_team_id)
            return

        if achievement_team.status != AchievementTeam.STATUS_PENDING and not force:
            logger.error(_("AchievementTeam %s already processed") % achievement_team_id)
            return

        update_achievement_team_users(achievement_team=achievement_team, force=force)

        course_lesson_results = (
            CourseLessonResult.objects
            .filter(summary__clesson_id=achievement_team.clesson_id, summary__student=achievement_team.captain)
        )
        for course_lesson_result in course_lesson_results:
            pull_triggers(course_lesson_result=course_lesson_result, force=True)

    except Exception:
        exc_info = traceback.format_exc()
        logger.error(exc_info)
        if achievement_team is not None:
            achievement_team.error_messages = str(exc_info)
            achievement_team.status = AchievementTeam.STATUS_ERROR
            achievement_team.save()


@logged_task
@app.task()
def grant_achievement_periodic_task():
    user_achievements = UserAchievement.objects.filter(published_at__isnull=True)
    for user_achievement in user_achievements:
        grant_achievement_task.delay(user_achievement_id=user_achievement.id)


@logged_task
@app.task()
def update_achievement_team_users_periodic_task(*args, **kwargs) -> None:
    achievement_teams = AchievementTeam.objects.filter(status=AchievementTeam.STATUS_PENDING)
    for achievement_team in achievement_teams:
        update_achievement_team_users_task.delay(achievement_team_id=achievement_team.id)
