import logging

from rest_framework.generics import CreateAPIView
from rest_framework.response import Response
from rest_framework.views import APIView

from kelvin.common.exceptions import APIExceptionWithStatusCode
from kelvin.common.yandex_forms.permissions import IsYandexFormsServicePermission

from .achievery_client import AchieveryHTTPError
from .achievery_client import client as achievery_client
from .serializers import (
    TeamAchievementYandexFormCreateSerializer, ValidateAchievementInputSerializer, ValidateAchievementOutputSerializer,
)

logger = logging.getLogger(__name__)


class ValidateAchievementView(APIView):
    input_serializer = ValidateAchievementInputSerializer
    output_serializer = ValidateAchievementOutputSerializer

    def post(self, request):

        serializer = self.input_serializer(data=self.request.data)
        serializer.is_valid(raise_exception=True)

        achievement_id = serializer.validated_data['achievement_id']
        level = serializer.validated_data['level']
        try:
            resp = achievery_client.check_achievement(
                achievement_id=achievement_id,
                level=level,
                fields=['id'],
            )
            return Response(self.output_serializer(resp).data)

        except AchieveryHTTPError as ex:
            raise APIExceptionWithStatusCode(
                detail={'achievery_api_response': ex.response.text},
                code=ex.response.reason,
                status_code=ex.response.status_code
            )


class YandexFormTeamAchievementView(CreateAPIView):
    permission_classes = [
        IsYandexFormsServicePermission,
    ]
    authentication_classes = []
    serializer_class = TeamAchievementYandexFormCreateSerializer
