from builtins import map, object
from operator import attrgetter, itemgetter

from django import forms
from django.conf.urls import url
from django.contrib import admin
from django.contrib.auth import get_user_model
from django.core.exceptions import PermissionDenied
from django.core.urlresolvers import reverse
from django.http import Http404, HttpResponseRedirect
from django.utils.translation import ugettext_lazy as _

from kelvin.certificates.models import CertificateContext, CertificateTemplate, RenderedCertificate

User = get_user_model()


class CertificateTemplateAdmin(admin.ModelAdmin):
    """Админка шаблона сертификата"""
    list_display = ('id', 'name',)


class CertificateContextForm(forms.ModelForm):
    """Форма контекста сертификата"""
    class Meta(object):
        model = CertificateContext
        fields = '__all__'

    @staticmethod
    def user_label(student):
        """Представление ученика для формы"""
        return '{0}: {1} ({2})'.format(
            student.username, student.get_full_name(), student.id,
        )

    def __init__(self, *args, **kwargs):
        """
        Дает выбирать учеников группы курса в поле 'students', либо учеников
        всех курсов-копий, если курс оригинальный
        """
        super(CertificateContextForm, self).__init__(*args, **kwargs)
        student_choices = set(
            map(itemgetter(1), self.instance.students_from_course()))

        self.fields['students'].widget.choices = [
            (student.id, self.user_label(student))
            for student in sorted(student_choices, key=attrgetter('id'))
        ]

    def clean(self):
        """Если изменился курс, то нужно очистить список выбранных учеников"""
        super(CertificateContextForm, self).clean()
        if self.instance.pk and (
                self.instance.course != self.cleaned_data['course']):
            self.cleaned_data['students'] = []


class CertificateContextAdmin(admin.ModelAdmin):
    """Админка контекста сертификата"""
    list_display = ('id', 'name', 'template', 'course')
    raw_id_fields = ('template', 'course')
    filter_horizontal = ('students',)

    form = CertificateContextForm

    change_form_template = (
        'certificates/admin/certificatecontext_change_form.html'
    )

    def get_queryset(self, request):
        """
        Дополнительно подтягиваем шаблон, курс и группу курса
        """
        qs = super(CertificateContextAdmin, self).get_queryset(request)
        return qs.select_related('template', 'course')

    def get_urls(self):
        """
        Добавляет урл (пере)генерации всех сертификатов по данному контексту
        """
        return [
            url(r'(?P<object_id>\d+)/generate_certificates/$',
                self.admin_site.admin_view(self.generate_all),
                name='certificate_certificatecontext_generate_all')
        ] + super(CertificateContextAdmin, self).get_urls()

    def generate_all(self, request, **kwargs):
        """
        (Пере)генерация всех сертификатов из данного контекста
        """
        if not self.has_change_permission(request):
            raise PermissionDenied

        cert_ctx = self.get_object(request, kwargs['object_id'])
        if cert_ctx is None:
            raise Http404

        cert_ctx.generate_all()

        self.message_user(
            request,
            _('Генерация сертификатов "{0}" начата'.format(cert_ctx.name)),
        )
        return HttpResponseRedirect(
            reverse('admin:{0}_{1}_changelist'.format(
                CertificateContext._meta.app_label,
                RenderedCertificate._meta.model_name
            ), current_app=self.admin_site.name)
        )


class RenderedCertificateAdmin(admin.ModelAdmin):
    """Админка готовых сертификатов"""
    list_display = (
        'user',
        'get_course',
        'context',
        'date_updated',
    )
    list_select_related = ('user', 'context', 'context__course')
    raw_id_fields = ('user', 'context')
    search_fields = (
        'user__username',
        'user__first_name',
        'user__last_name',
        'context__course__id',
        'context__course__name',
    )

    change_form_template = (
        'certificates/admin/renderedcertificate_change_form.html'
    )

    def get_urls(self):
        """
        Добавляет урл повторного рендеринга
        """
        return [
            url(r'(?P<object_id>\d+)/rerender/$',
                self.admin_site.admin_view(self.rerender),
                name='certificate_renderedcertificate_rerender')
        ] + super(RenderedCertificateAdmin, self).get_urls()

    def get_course(self, obj):
        """Название курса"""
        return obj.context.course.name
    get_course.short_description = 'Курс'
    get_course.admin_order_field = 'context__course__name'

    def rerender(self, request, **kwargs):
        """
        Запускает повторный рендеринг данного сертификата.
        """
        if not self.has_change_permission(request):
            raise PermissionDenied

        cert = self.get_object(request, kwargs['object_id'])
        if cert is None:
            raise Http404

        cert.rerender()

        self.message_user(
            request,
            _('Перегенерация сертификата {0} запущена'.format(cert.id)),
        )
        return HttpResponseRedirect(
            reverse(
                'admin:{0}_{1}_change'.format(
                    RenderedCertificate._meta.app_label,
                    RenderedCertificate._meta.model_name
                ),
                current_app=self.admin_site.name,
                args=(kwargs['object_id'],),
            )
        )


admin.site.register(CertificateTemplate, CertificateTemplateAdmin)
admin.site.register(CertificateContext, CertificateContextAdmin)
admin.site.register(RenderedCertificate, RenderedCertificateAdmin)
