from builtins import map, object

from django.conf import settings
from django.db import models
from django.template import Context, engines
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import TimeStampMixin
from kelvin.courses.models import Course


@python_2_unicode_compatible
class CertificateTemplate(models.Model):
    """
    Шаблон сертификата - HTML с джанго-темплейтами
    """
    name = models.CharField(
        verbose_name=_('Название шаблона'),
        max_length=255,
    )
    template = models.TextField(
        verbose_name=_('Разметка шаблона'),
    )

    class Meta(object):
        verbose_name = _('Шаблон сертификата')
        verbose_name_plural = _('Шаблоны сертификатов')

    def __str__(self):
        """Строковое отображение"""
        return '{0} ({1})'.format(self.name, self.id)

    def render(self, context):
        """
        Рендеринг шаблона с использованием переданного контекста
        средствами стандартного django-шаблонизатора
        """
        return engines['django'].from_string(
            self.template).render(Context(context))


@python_2_unicode_compatible
class CertificateContext(models.Model):
    """
    Связь шаблона и информации, нужной для его рендеринга
    """
    template = models.ForeignKey(
        CertificateTemplate,
        verbose_name=_('Шаблон сертификата'),
    )
    name = models.CharField(
        verbose_name=_('Название сертификата'),
        max_length=255,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_('Курс, к которому относится сертификат'),
    )
    students = models.ManyToManyField(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Ученики, которым нужно сгенерировать сертификат'),
        blank=True,
    )
    additional_context = JSONField(
        verbose_name=_('Дополнительные параметры рендеринга'),
        blank=True,
        default={},
    )

    class Meta(object):
        verbose_name = _('Информация для генерации сертификата')
        verbose_name_plural = _('Информация для генерации сертификатов')

    def __str__(self):
        """Строковое отображение"""
        return 'Курс id {0}, шаблон id {1}'.format(
            self.course_id, self.template_id)

    def generate_all(self):
        """
        Генерирует все сертификаты из данного контекста
        """
        from kelvin.certificates.tasks import (
            generate_certificates_from_context,
        )

        generate_certificates_from_context.delay(self.id)

    def students_from_course(self):
        """
        Возвращает всех студентов из курса с указанием,
        какому курсу студент соответствует.
        """
        def make_annotator(course):
            return lambda student: (course, student)
        students = []

        if self.id and self.course_id:
            course = self.course
            if not course.copy_of:
                for copy in Course.objects.filter(
                        copy_of=course).prefetch_related('students'):
                    students.extend(list(map(
                        make_annotator(copy),
                        copy.students.all())))
            else:
                students.extend(list(map(
                    make_annotator(course),
                    course.students.all()))
                )

        return students


class RenderedCertificate(TimeStampMixin, models.Model):
    """
    Отрендеренный сертификат пользователя
    """
    user = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Пользователь'),
    )
    html_text = models.TextField(
        verbose_name=_('html-текст сертификата'),
        blank=True,
    )
    file = models.FileField(
        verbose_name=_('Готовый сертификат'),
        upload_to='course_certificates',
        blank=True,
        null=True,
    )
    context = models.ForeignKey(
        CertificateContext,
        verbose_name=_('Контекст сертификата'),
    )

    class Meta(object):
        verbose_name = _('Готовый сертификат')
        verbose_name_plural = _('Готовые сертификаты')

    def rerender(self):
        """
        (Пере)генерирует PDF для текущего сертификата
        """
        from kelvin.certificates.tasks import rerender_certificate

        rerender_certificate.delay(self.id)
