import hashlib
import logging

from django.core.files.base import ContentFile

from kelvin.celery import app
from kelvin.certificates.models import CertificateContext, RenderedCertificate

logger = logging.getLogger(__name__)


@app.task()
def generate_certificates_from_context(cert_ctx_id):
    """
    Генерирует сертификаты из контекста
    """
    # не ловим исключение, увидим зафейленный таск в селери-админке
    cert_ctx = CertificateContext.objects.select_related(
        'course',
        'template',
    ).prefetch_related(
        'students',
    ).get(id=cert_ctx_id)

    template = cert_ctx.template

    course = cert_ctx.course

    if cert_ctx.students.exists():
        students = cert_ctx.students.all()
    else:
        students = cert_ctx.students_from_course()

    for student in students:
        if isinstance(student, tuple):
            course, student = student

        certificate, _ = RenderedCertificate.objects.get_or_create(
            user=student,
            context=cert_ctx,
        )
        certificate.html_text = template.render({
            'course': course,
            'student': student,
            'context': cert_ctx,
            'extra': cert_ctx.additional_context,
        })
        certificate.save()
        certificate.rerender()


@app.task()
def rerender_certificate(cert_id):
    """
    Перегенерировать сертификат
    """
    # не ловим исключение, увидим зафейленный таск в селери-админке
    certificate = RenderedCertificate.objects.get(id=cert_id)
    # FIXME: починить зависимости
    pdf_data = pdfize(certificate.html_text)

    if not pdf_data:
        raise RuntimeError(
            'error generating Certificate with id: {0}'.format(cert_id),
        )

    filename = 'certificate_{ctx_id}_{cert_id}_{user_id}_{hash}.pdf'.format(
        cert_id=cert_id,
        ctx_id=certificate.context_id,
        user_id=certificate.user_id,
        hash=hashlib.md5(pdf_data).hexdigest()[:7].upper()
    )
    certificate.file.delete(save=False)
    certificate.file.save(filename, ContentFile(pdf_data))
