from rest_condition import Or

from django.contrib.auth import get_user_model

from rest_framework.decorators import list_route
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_403_FORBIDDEN
from rest_framework.viewsets import GenericViewSet

from kelvin.accounts.permissions import IsParent, IsStaff, IsTeacher
from kelvin.certificates.filters import RenderedCertificateFilterSet
from kelvin.certificates.models import CertificateContext, RenderedCertificate
from kelvin.certificates.serializers import RenderedCertificateSerializer
from kelvin.courses.models import Course

User = get_user_model()


class RenderedCertificateViewSet(GenericViewSet):
    """Вьюсет готовых сертификатов"""
    queryset = RenderedCertificate.objects.select_related('context')
    serializer_class = RenderedCertificateSerializer
    filter_class = RenderedCertificateFilterSet

    @list_route(permission_classes=[
        IsAuthenticated,
        IsStaff,  # FIXME temporary permission
    ])
    def my(self, request, *args, **kwargs):
        """
        Возвращает сертификаты пользователя. При запросе с `course_id`
        возвращает только сертификаты по этому курсу
        """
        user_certificates = self.filter_queryset(
            self.get_queryset()).filter(user=request.user)
        serializer = self.get_serializer(many=True)

        return Response(serializer.to_representation(user_certificates))

    @list_route(permission_classes=[
        IsAuthenticated, Or(IsTeacher, IsStaff, IsParent)])
    def student(self, request, *args, **kwargs):
        """
        Возвращает сертификаты ученика `student_id`, если пользователь является
        его учителем, его родителем или администратором системы. При запросе с
        `course_id` возвращает только сертификаты по этому курсу
        """
        student_id = request.query_params.get('student_id')
        if student_id is None:
            return Response('`student_id` parameter is required',
                            status=HTTP_400_BAD_REQUEST)

        user_is_parent_qs = User.objects.filter(
            id=self.request.user.id,
            parent_profile__children__id=student_id,
        )
        user_is_teacher_qs = Course.objects.filter(
            owner=self.request.user,
            students__id=student_id,
        ).values_list('id', flat=True)

        if not (self.request.user.is_parent and user_is_parent_qs.exists() or
                self.request.user.is_teacher and user_is_teacher_qs or
                self.request.user.is_staff):
            return Response('Access denied for this student',
                            status=HTTP_403_FORBIDDEN)

        user_certificates = self.filter_queryset(
            self.get_queryset()).filter(user_id=student_id)

        # Не даем учителю сертификаты не по его курсам
        if not request.user.is_staff and request.user.is_teacher:
            user_certificates = user_certificates.filter(
                context__course_id__in=user_is_teacher_qs
            )

        serializer = self.get_serializer(many=True)

        return Response(serializer.to_representation(user_certificates))
