from builtins import object
from functools import wraps

from codemirror2.widgets import AdminCodeMirrorEditor

from django import forms
from django.contrib import admin
from django.contrib.admin import AllValuesFieldListFilter, site, widgets
from django.contrib.auth import get_user_model
from django.core.exceptions import PermissionDenied
from django.db.models import Q
from django.template.loader import render_to_string
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _

from kelvin.common.form_fields import JSONFormField

User = get_user_model()


class DropDownFilter(AllValuesFieldListFilter):
    """
    Выпадающий фильтр для списков в админке
    """
    template = 'common/admin/dropdown_filter.html'
    display_title = ''

    def __init__(self, *args, **kwargs):
        """
        Переопределяет название фильтра
        """
        super(DropDownFilter, self).__init__(*args, **kwargs)
        self.title = self.display_title or self.title


class ChoicesDropDownFilter(DropDownFilter):
    """
    Фильтр, который обрабатывает в `lookup_choices` список кортежей из
    значения и отображаемого значения
    """

    def choices(self, obj):
        """
        Переопределяет choices, чтобы изменить отображаемое значение
        """
        yield {
            'selected': (self.lookup_val is None and self.lookup_val_isnull is None),
            'query_string': obj.get_query_string(
                {}, [self.lookup_kwarg, self.lookup_kwarg_isnull]),
            'display': _('All'),
        }
        include_none = False
        for val, text in self.lookup_choices:
            if val is None:
                include_none = True
                continue
            yield {
                'selected': self.lookup_val == val,
                'query_string': obj.get_query_string({
                    self.lookup_kwarg: val,
                }, [self.lookup_kwarg_isnull]),
                'display': text,
            }
        if include_none:
            yield {
                'selected': bool(self.lookup_val_isnull),
                'query_string': obj.get_query_string({
                    self.lookup_kwarg_isnull: 'True',
                }, [self.lookup_kwarg]),
                'display': self.empty_value_display,
            }


def build_dropdown_filter_with_title(title):
    """
    Генерирует класс-наследник `DropDownFilter` с переданным `title`
    """
    class NewClass(DropDownFilter):
        display_title = title
    return NewClass


class TeacherForeignKeyWidget(widgets.ForeignKeyRawIdWidget):
    """
    Widget для поля `ForeignKey` к модели `User` с фильтром по учителям
    """

    def url_parameters(self):
        res = super(TeacherForeignKeyWidget, self).url_parameters()
        res['is_teacher__exact'] = 1
        return res


class ContentManagerForeignKeyWidget(widgets.ForeignKeyRawIdWidget):
    """
    Widget для поля `ForeignKey` к модели `User` с фильтром по
    контент-менеджерам
    """

    def url_parameters(self):
        res = super(ContentManagerForeignKeyWidget, self).url_parameters()
        res['is_content_manager__exact'] = 1
        return res


class OwnerOnlyTeacherModelFormMixin(object):
    """
    Mixin, позваляющий делать поиск по полю owner в админке и добавляющий
    фильтр только по учителям
    """

    def __init__(self, *args, **kwargs):
        assert isinstance(self, forms.ModelForm)
        super(OwnerOnlyTeacherModelFormMixin, self).__init__(*args, **kwargs)
        owner_field = self.fields.get('owner', None)
        if owner_field:
            owner_field.queryset = User.objects.filter(is_teacher=True)
            owner_field.widget = TeacherForeignKeyWidget(
                rel=self._meta.model._meta.get_field('owner').rel, admin_site=site)


class OwnerOnlyContentManagerModelFormMixin(object):
    """
    Фильтр только по контент-менеджерам для поля 'owner' формы
    """

    def __init__(self, *args, **kwargs):
        assert isinstance(self, forms.ModelForm)
        super(OwnerOnlyContentManagerModelFormMixin, self).__init__(
            *args, **kwargs)
        self.fields['owner'].queryset = User.objects.filter(
            is_content_manager=True)
        self.fields['owner'].widget = ContentManagerForeignKeyWidget(
            rel=self._meta.model._meta.get_field('owner').rel, admin_site=site)


class LabellessRawIdWidget(widgets.ForeignKeyRawIdWidget):
    """
    Убирает метку у поля ввода, чтобы не делать лишних запросов
    """

    def label_for_value(self, value):
        return ''


class LabellessRawIdFieldsMixin(object):
    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        """
        Переопределяет виджет для поля занятия
        """
        if db_field.name in self.labelless_raw_id_fields:
            db = kwargs.get('using')
            kwargs['widget'] = LabellessRawIdWidget(
                db_field.rel, self.admin_site, using=db
            )

            if 'queryset' not in kwargs:
                queryset = self.get_field_queryset(db, db_field, request)
                if queryset is not None:
                    kwargs['queryset'] = queryset

            return db_field.formfield(**kwargs)

        return super(LabellessRawIdFieldsMixin, self).formfield_for_foreignkey(
            db_field, request=request, **kwargs)


class JsonValidatingTextarea(widgets.AdminTextareaWidget):
    """
    Виджет текстового поля с кнопкой, валидирующей json в поле
    с помощью javascript
    """

    def render(self, name, value, attrs=None):
        """
        Добавляет кнопку для json-валидации, а также поле для вывода ошибок
        """
        textarea_html = super(widgets.AdminTextareaWidget, self).render(
            name, value, attrs)
        return mark_safe(render_to_string(
            'common/widgets/json_validating_widget.html',
            {'textarea_html': textarea_html, 'textarea_id': attrs['id']},
        ))


class InlineJSONFormField(JSONFormField):
    """
    Поле для JSON отображаемое в одну строчку
    """

    def __init__(self, *args, **kwargs):
        super(InlineJSONFormField, self).__init__(*args, **kwargs)

        self.inline = True

    def widget_attrs(self, widget):
        """
        Изменяем атрибуты размера TextArea
        """
        attrs = super(InlineJSONFormField, self).widget_attrs(widget)
        attrs.update({'rows': '1', 'cols': '20'})
        return attrs


class AdminJSONEditWidget(AdminCodeMirrorEditor):
    """
    Виджет для JSON-полей в админке: добавляет подсветку синтаксиса,
    моноширный шрифт и удобство редактирования.

    Можно использовать через formfield_overrides:

    class SomeThing(admin.ModelAdmin):
        formfield_overrides = {
            JSONField: {'widget': AdminJSONEditWidget}
        }
    """
    default_options = {
        'mode': 'javascript',
        'theme': 'eclipse',
        'smartIndent': False
    }
    default_template = 'common/widgets/codemirror_widget.html'

    def __init__(self, *args, **kwargs):
        kwargs.setdefault('options', self.default_options.copy())
        kwargs.setdefault('script_template', self.default_template)
        super(AdminJSONEditWidget, self).__init__(*args, **kwargs)


class UserBlameModelAdminMixin(admin.ModelAdmin):
    def save_model(self, request, obj, form, change):
        if obj.created_by:
            obj.modified_by = request.user
        else:
            obj.created_by = request.user

        return super().save_model(request, obj, form, change)


class AvailableForSupportAdminMixinBase:
    def get_queryset(self, request):
        qs = super().get_queryset(request)
        user = request.user
        if user and not user.is_anonymous and user.is_authenticated and user.is_support:
            qs = qs.filter(
                Q(available_for_support=True) |
                Q(created_by=user)
            )

        return qs

    def get_readonly_fields(self, request, obj=None):
        ro_fields = super().get_readonly_fields(request, obj)
        if obj is not None:
            user = request.user
            if user and not user.is_anonymous and user.is_authenticated and user.is_support:
                ro_fields = list(
                    set(ro_fields) |
                    {
                        'available_for_support',
                    }
                )

        return ro_fields


class AvailableForSupportAdminMixin(AvailableForSupportAdminMixinBase, admin.ModelAdmin):
    pass


class AvailableForSupportInlineAdminMixin(AvailableForSupportAdminMixinBase, admin.options.InlineModelAdmin):
    pass


def superuser_only_action(func):
    @wraps(func)
    def inner(self, request, *args, **kwargs):
        if request.user and request.user.is_superuser:
            return func(self, request, *args, **kwargs)
        else:
            raise PermissionDenied(_('For superusers only'))

    return inner
