import functools
import hashlib
import json
import logging
import random
import time

from celery import task
from celery.exceptions import InvalidTaskError
from celery.utils.log import get_task_logger

from django.conf import settings

from kelvin.common.redis import redis

logger = get_task_logger(__name__)
throttler_logger = logging.getLogger('celery_throttler')


def with_inaccuracy(number, inaccuracy=None):
    """
    Add inaccuracy for number (x).
    How it work:
    - for inaccuracy=0 return just x
    - for inaccuracy=0.1(10%) return random number from 0.9x to 1.1x
    - for inaccuracy=0.5(50%) return random number from 0.5x to 1.5x
    """
    if inaccuracy is None:
        inaccuracy = settings.THROTTLED_TASK_INACCURACY
    term, factor = 1 - inaccuracy, random.random() * inaccuracy * 2
    return int(round(number * (factor + term)))


def default_key_generator(*args, **kwargs):
    """
    Simple key generator for task calls.

    Make json dump from params and calculate hash.
    You need a custom generator if your task have non-serializable params!
    """
    try:
        # https://docs.python.org/3/reference/datamodel.html#object.__hash__
        # Changed in version 3.3: Hash randomization is enabled by default.
        return hashlib.sha256(json.dumps([args, kwargs], sort_keys=True).encode('utf-8')).hexdigest()
    except Exception:
        raise InvalidTaskError(
            'Dafault key generator can`t generate key for this arguments!'
        )


def throttled_task(generator=default_key_generator, *task_args, **task_kwargs):
    """
    The decorator starts the pending task and drop flag in redis.

    :param generator - function, takes args & kwargs from the task and returns
    the same string value for calls that can be collocated.
    """
    def decorator(func):
        @_deferred_task(generator)
        @task(*task_args, **task_kwargs)
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            if 'throttled_task_key' not in kwargs:
                logger.info('[throttled_task] call task from __call__()')
                return func(*args, **kwargs)

            # Remove task`s flag in redis
            key = kwargs.pop('throttled_task_key')
            redis.client.delete(key)
            logger.info('[throttled_task] worker run task #{}'.format(key))
            return func(*args, **kwargs)

        return wrapper

    return decorator


def _deferred_task(key_generator):
    """
    The decorator defers the launch of the app.task to a specific time and
    stores in Redis flag of task for throttling other identical tasks.

    :param generator - function, takes args & kwargs from the task and returns
    the same string value for calls that can be collocated.
    """

    def monkey_patcher(task):
        original_apply_async = task.apply_async

        def custom_apply_async(args=None, kwargs=None, *task_args, **options):
            if 'throttled_task_key' in kwargs:
                # If somebody will use reserved kwarg in own tasks
                throttler_logger.error(
                    '[throttled_task] kwarg "throttled_task_key" is reserved'
                )
                raise InvalidTaskError(
                    'kwarg "throttled_task_key" is reserved by Throttler',
                )

            key = '{}.{}'.format(task.name, key_generator(*args, **kwargs))

            # Give countdown from settings and little randomize
            countdown_original = settings.THROTTLED_TASK_TIMES[task.name]
            countdown = with_inaccuracy(countdown_original)

            options['countdown'] = options.get('countdown', countdown)

            stringified_timestamp = redis.client.get(key)

            try:
                parsed_timestamp = float(stringified_timestamp)
                time_spent = time.time() - parsed_timestamp
            except TypeError as ValueError:
                parsed_timestamp = None
                time_spent = None

            # if key does not exist or it is outdated - just start planning
            if parsed_timestamp is None or time_spent > options['countdown']:
                redis.client.setex(name=key, time=options['countdown'], value=time.time())
            else:
                return throttler_logger.info(
                    '[throttled_task] throttled #{}'.format(key),
                )

            throttler_logger.info(
                '[throttled_task] planned for {} sec #{}'.format(
                    options['countdown'], key,
                ),
            )
            kwargs['throttled_task_key'] = key
            return original_apply_async(args, kwargs, *task_args, **options)

        def custom_delay(*args, **kwargs):
            """
            We need replace it because original delay don't call custom
            apply_async. Else it patch just in celery`s proxy.
            """

            return custom_apply_async(args, kwargs)

        task.delay = custom_delay
        task.apply_async = custom_apply_async
        return task

    return monkey_patcher
