"""
Функционал позволяет в случае любого исключения или ошибки вернуть
клиенту application/json ответ со содержимым следующего формата:

{
    "errors": [
        {
            'source': '<источник происхождения ошибки>',
            'code': '<человеко-читаемый текстовый код ошибки>,
            'message': '<текстовое сообщение об ошибке>',
        {
            ...
        }
    ]
}

Где:

    * source – источник происхождения ошибки, это может быть название
        поля в форме или сериализаторе, либо "non_field_errors", если
        сообщение не относится к какому-либо полю, но имеет отношение
        к валидации формы, либо "__response__" для всех остальных ошибок.
    * code – человеко-читаемый код ошибки, должен состоять из символов
        a-zA-Z и подчеркивания, например "not_found".
    * message – текстовое сообщение, раскрывающее причину ошибки,
        например "Вы не авторизованы".

Если вы хотите использовать стороннюю локализацию сообщений об ошибках,
лучше использовать в качестве ключа пару <source>.<code>.
"""
import json
import logging
from builtins import object
from collections import Mapping

from past.builtins import basestring

from django.core.exceptions import PermissionDenied as DjangoPermissionDenied
from django.core.exceptions import ValidationError as DjangoValidationError
from django.http import Http404, HttpResponseServerError
from django.utils.encoding import smart_str
from django.utils.translation import ugettext_lazy as _

from rest_framework.exceptions import APIException, AuthenticationFailed, NotAuthenticated, NotFound, PermissionDenied
from rest_framework.fields import get_error_detail
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_401_UNAUTHORIZED, HTTP_403_FORBIDDEN, HTTP_404_NOT_FOUND
from rest_framework.views import exception_handler as drf_exception_handler

logger = logging.getLogger(__name__)


class ErrorsComposer(object):
    """
    Получает на вход экземпляр исключения и транформирует его
    в необходимый формат для отображения ошибок.

    Неподдерживаемые исключения не обрабатываются.
    """
    default_source = '__response__'
    default_code = 'unknown'
    default_message = _('Неизвестная ошибка')

    def __init__(self, exception):
        self.exception = exception
        self.exception_supported = None
        self.status_code = HTTP_400_BAD_REQUEST
        self.details = None
        self._set_defaults()

    def _set_defaults(self):
        """
        В зависимости от типа исключения выставляем дефолтные
        сообщение, код ошибки и код ответа.
        """
        self.exception_supported = True

        if isinstance(self.exception, DjangoValidationError):
            self.details = get_error_detail(self.exception)

        elif isinstance(self.exception, APIException):
            self.default_code = self.exception.default_code
            self.default_message = self.exception.default_detail
            self.details = self.exception.get_full_details()
            self.status_code = self.exception.status_code

        elif isinstance(self.exception, Http404):
            self.default_code = NotFound.default_code
            self.default_message = (
                smart_str(self.exception) or NotFound.default_detail
            )
            self.status_code = HTTP_404_NOT_FOUND

        elif isinstance(self.exception, DjangoPermissionDenied):
            self.default_code = PermissionDenied.default_code
            self.default_message = (
                smart_str(self.exception) or
                PermissionDenied.default_detail
            )
            self.status_code = HTTP_403_FORBIDDEN
        else:
            self.exception_supported = False

    def compose_error(self, source=None, message=None, code=None):
        return {
            'source': source or self.default_source,
            'code': self.replace_spaces(
                code or getattr(message, 'code', self.default_code)
            ),
            'message': message or self.default_message,
        }

    @staticmethod
    def replace_spaces(code):
        """
        Заменяем в коде пробелы на поддеркивания.
        """
        if isinstance(code, basestring):
            return code.replace(' ', '_')
        return code

    def get_representation(self):
        """
        В зависимости от переданной в конструктор исключения, возвращает
        список ошибок в специальном формате:

        [
            {
                'source': '<источник происхождения ошибки>',
                'code': '<человеко-читаемый текстовый код ошибки>,
                'message': '<текстовое сообщение об ошибке>',
            },
            ...
        ]

        :rtype list:
        :returns: список словарей с ошибками
        """
        if not self.details:
            return [self.compose_error()]

        return self.parse_details(self.details)

    def parse_details(self, details, source=None):
        """
        Парсим (иногда рекурсивно) содержимое исключения, чтобы
        сформировать список сообщений об ошибках.
        """
        errors = []
        source = source or self.default_source

        if isinstance(details, Mapping):
            # Возможные варианты:
            # {field: [{code: ..., message...}, {code: ..., message: ...}]}
            # {field: [message, message]}
            # {field: message}
            # {code: ..., message...}
            if set(details) == {'code', 'message'}:
                errors.append(
                    self.compose_error(
                        source=source,
                        message=details['message'],
                        code=details['code'],
                    )
                )
            else:
                for field, messages in details.items():
                    errors.extend(self.parse_details(messages, field))

        elif isinstance(details, (list, tuple)):
            # Возможные варианты:
            # [{code: ..., message...}, {code: ..., message: ...}]
            # [message, message]
            for message in details:
                if isinstance(message, Mapping):
                    errors.append(
                        self.compose_error(
                            source=source,
                            message=message.get('message'),
                            code=message.get('code'),
                        )
                    )
                else:
                    errors.extend(self.parse_details(message, source))

        elif isinstance(details, basestring):
            # Возможные варианты:
            # message
            errors.append(self.compose_error(source=source, message=details))
        else:
            logger.warning(
                'Unsupported "details" type %r', details,
                extra={'data': {'details': details, 'exc': self.exception}}
            )

        return errors

    def get_response_body(self):
        """
        Генерирует тело ответа.
        """
        return {
            'errors': self.get_representation()
        }

    @classmethod
    def compose_response_body(cls, exception=None, source=None,
                              message=None, code=None):
        """
        Метод для генерации кастомизированного тела ответа.
        """
        composer = cls(exception)
        if source:
            composer.default_source = source
        if message:
            composer.default_message = message
        if code:
            composer.default_code = code

        return composer.get_response_body()


def exception_handler(exc, context):
    """
    Глобальный хендлер исключений.
    """
    if isinstance(exc, (NotAuthenticated, AuthenticationFailed)):
        exc.status_code = HTTP_401_UNAUTHORIZED

    composer = ErrorsComposer(exc)
    response = drf_exception_handler(exc, context)

    if not composer.exception_supported:
        # не перехватываем generic эксепшены, иначе
        # сломается вся логика, которая их обрабатывает позже:
        # сигналы, Sentry, мидлевари и т.д.
        return response

    response_body = composer.get_response_body()

    if response is None:
        response = Response(response_body, status=composer.status_code)
    else:
        response.data = response_body

    return response


# прекешируем ответ, т.к. на 500-ку он будет всегда один и тот же
SERVER_ERROR_RESPONSE = HttpResponseServerError(
    smart_str(
        json.dumps(
            ErrorsComposer(APIException()).get_response_body(),
            ensure_ascii=False,
        )
    )
)


def server_error(request):
    """
    Глобальный обработчик для unhandled excepions.
    """
    return SERVER_ERROR_RESPONSE
