from copy import deepcopy

import jsonschema

from django import forms
from django.core.exceptions import ValidationError

from .model_mixins import INFO_SCHEMA


class InfoFormMixin(forms.ModelForm):
    """
    Миксин добавляющий в форму поле info
    """
    help_text_template = u"""Пример json-а:
    {{
        "about": {{
            "header": "Замечательный курс",
            "text": "О курсе"
        }},
        "authors": [
            {{
                "photo": "url_with_picture",
                "fio": "Иваныч",
                "about": "Физрук"
            }}
        ]{}
    }}
    """

    def __init__(self, *args, **kwargs):
        super(InfoFormMixin, self).__init__(*args, **kwargs)

        self.additional_info_props = getattr(self, 'additional_info_props', [])
        self.required_info_props = getattr(self, 'required_info_props', [])

        if 'info' in self.fields:
            self.fields['info'].help_text = self.get_help_text()

    def get_help_text(self):
        """
        Расширяет help_text для информации
        берет tuple полей из self.additional_info_props
        и сопоставляет их с методами info_<field>_help_data
        добавляя в help_text
        Если определен шаблон в self.info_help_text_template,
        то берется он, иначе дефолтный шаблон
        """

        help_text_data = ''

        if self.additional_info_props:
            for prop in self.additional_info_props:
                if not hasattr(self, 'info_{}_help_text'.format(prop)):
                    continue
                help_text = getattr(self, 'info_{}_help_text'.format(prop))()
                help_text_data += ',\n\t"{}": {}'.format(prop, help_text)

        help_text_template = getattr(
            self, 'info_help_text_template', self.help_text_template)

        return help_text_template.format(help_text_data)

    def get_info_schema(self):
        """
        Расширяет схему для информации
        берет tuple полей из self.additional_info_props
        и сопоставляет их с методами info_<field>_schema
        добавляя в схему
        Также добавляет поля из self.required_info_props
        в schema['required']
        """
        if not self.additional_info_props:
            return INFO_SCHEMA

        schema = deepcopy(INFO_SCHEMA)

        if self.additional_info_props:
            for prop in self.additional_info_props:
                if not hasattr(self, 'info_{}_schema'.format(prop)):
                    continue
                schema['properties'][prop] = getattr(
                    self, 'info_{}_schema'.format(prop)
                )()

        if self.required_info_props:
            schema['required'] += self.required_info_props

        return schema

    def clean_info(self):
        """
        Проверка поля по схеме
        """
        if self.cleaned_data['info'] is not None:
            try:
                jsonschema.validate(self.cleaned_data['info'],
                                    self.get_info_schema())
            except jsonschema.ValidationError as e:
                raise ValidationError(e.message)
        return self.cleaned_data['info']
