import json
from typing import Union

import kikimr.public.sdk.python.persqueue.auth as auth
import kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api as pqlib

from django.conf import settings


class LogbrokerClient:
    """Клиент для логброкера.
    with LogbrokerClient() as logbroker:
        ...
    """
    timeout: int = 15

    def __init__(self, *, token: str = '', datacenter: str = None, timeout: int = None):
        """
        :param token: Токен доступа.
        :param datacenter: man, vla, myt ...
        :param timeout:
        """
        if datacenter is None:
            datacenter = settings.LOGBROKER_DATACENTER
        self._api = None
        self.timeout = timeout or self.timeout
        self.endpoint = f'{datacenter}.logbroker.yandex.net'

        if not token:
            raise ValueError("Please supply Logbroker access token for LogbrokerClient.")

        self.credentials = auth.OAuthTokenCredentialsProvider(token.encode())

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.api.quit()

    @property
    def api(self):
        if self._api is None:
            self._api = pqlib.PQStreamingAPI(self.endpoint, settings.LOGBROKER_PORT)
            api_start_future = self._api.start()
            api_start_future.result(timeout=self.timeout)

        return self._api

    def get_producer(self, *, source: str, topic: str):
        """Возвращает писателя в определённый раздел.
        :param source: Идентификатор, описывающий источник. Произвольная строка.
        :param topic: Раздел, куда требуется производить запись.
        """
        return LogbrokerProducer(client=self, source=source, topic=topic)


class LogbrokerProducer:
    """Писчий в логброкер.
    with LogbrokerClient() as logbroker:
        with logbroker.get_producer(source='me', topic='some/there/here') as producer:
            producer.write({'some': 'value'})
            producer.write('plaintext')
    """
    timeout: int = 10

    def __init__(self, *, client: 'LogbrokerClient', source: str, topic: str, timeout: int = None):

        self.timeout = timeout or self.timeout
        self.client = client
        self.seq_num = 0
        self.topic = topic
        self.source = source
        self._producer = None

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.producer.stop()

    @property
    def producer(self):
        if self._producer is None:
            client = self.client

            configurator = pqlib.ProducerConfigurator(self.topic.encode(), self.source.encode())
            producer = client.api.create_producer(configurator, credentials_provider=client.credentials)

            start_future = producer.start()
            result = start_future.result(timeout=self.timeout)

            if not result.HasField('init'):
                raise AssertionError(f'Unable to initialize Logbroker producer: {result}')

            self.seq_num = result.init.max_seq_no
            self._producer = producer

        return self._producer

    def write(self, message: Union[str, dict, bytes]):
        producer = self.producer

        self.seq_num += 1

        seq_num = self.seq_num

        if isinstance(message, dict):
            message = json.dumps(message)

        if isinstance(message, str):
            message = message.encode()

        response = producer.write(seq_num, message)

        result = response.result(timeout=self.timeout)

        if not result.HasField('ack'):
            raise AssertionError(f'Unable to write Logbroker message: {result}')

        return seq_num


def send_me(event: dict, source: str, topic: str, token: str):
    with LogbrokerClient(token=token) as logbroker:
        with logbroker.get_producer(source=source, topic=topic) as producer:
            event['source'] = source
            event['dc'] = settings.DEPLOY_NODE_DC
            producer.write(event)
