from builtins import object

import requests
from six.moves.urllib.parse import urlparse

from django.conf import settings

from kelvin.common.tvm import get_service_ticket


class MatchmanError(Exception):
    pass


class Matchman(object):
    """Клиент для запросов в Matchman микросервис."""

    def __init__(self, url):
        self.url = url

    def request(self, path):
        """Отправка запроса в Matchman."""
        try:
            resp = requests.get(
                url=urlparse.urljoin(self.url, path),
                verify=False,
                timeout=(settings.MATCHMAN_CONNECTION_TIMEOUT, settings.MATCHMAN_READ_TIMEOUT),
                headers={
                    "x-tvm2-ticket": get_service_ticket(settings.MATCHMAN_TVM_CLIENT_ID),
                },
            )
        except requests.ConnectionError as e:
            raise MatchmanError("Can't connect to matchman: {}".format(e))

        if resp.status_code != 200:
            raise MatchmanError('Unexpected status {}'.format(resp.status_code))

        if 'application/json' not in resp.headers['content-type']:
            raise MatchmanError('Response is not json')
        try:
            data = resp.json()
        except ValueError:
            raise MatchmanError('Response is not valid json: {}', resp.content)

        return data

    def get_user_courses(self, user_id):
        resp = self.request(
            path='match-courses/?user_id={user_id}'.format(
                user_id=user_id,
            )
        )
        try:
            matched_courses = resp['matched_courses']
        except KeyError:
            raise MatchmanError('Wrong response format for /match-courses: {}'.format(resp))
        return matched_courses

    def get_matched_rules(self, user_id):
        resp = self.request(
            path='match-rules/?user_id={user_id}'.format(
                user_id=user_id,
            )
        )
        try:
            matched_rules = resp['matched_rules']
        except KeyError:
            raise MatchmanError('Wrong response format for /match-rules: {}'.format(resp))
        return matched_rules

    def match_course(self, user_id, course_id):
        return self.request(
            path='match-course/?user_id={user_id}&course_id={course_id}'.format(
                user_id=user_id,
                course_id=course_id,
            )
        ).get('matched', False)


def create_matchman_client():
    """Создание клиента для запросов в Matchman"""
    client = Matchman(settings.MATCHMAN_URL)
    return client


matchman = create_matchman_client()  # Глобальный инстанс Matchman
