import json
import logging
from builtins import object
from datetime import datetime
from os import path

from django_yauth.user import DjangoUserDescriptor
from django_yauth.util import get_setting

from django.conf import settings
from django.contrib.auth import authenticate
from django.http.response import HttpResponse, HttpResponseForbidden

from rest_framework.response import Response

from kelvin.common.tvm import tvm2_client

logger = logging.getLogger(__name__)

try:
    from django.utils.deprecation import MiddlewareMixin
except ImportError:
    MiddlewareMixin = object


POSSIBLE_TVM2_HEADERS = ['HTTP_X_TVM2_TICKET', 'HTTP_X_YA_SERVICE_TICKET']




class BadRequestsLoggingMiddleware(MiddlewareMixin):
    """
    Middleware that logs error message if request has status HTTP 400
    """

    def process_response(self, request, response):
        if response.status_code == 400:
            # waiting for logging request body
            # https://github.com/tomchristie/django-rest-framework/issues/4050
            if isinstance(response, Response):
                logger.warning(
                    'HTTP 400 Bad Request, response: %s',
                    json.dumps(response.data),
                    extra={'request': request},
                )
            else:
                logger.warning(
                    'HTTP 400 Bad Request, response: %s',
                    response.content,
                    extra={'request': request},
                )
        return response


class TimeMiddleware(MiddlewareMixin):
    """
    Отдает время без запросов к базе, должна работать до `SessionMiddleware`
    и, в нашем случае, `AuthenticationMiddleware`
    """

    def process_request(self, request):
        if request.path == '/time/' and request.method == 'GET':
            if 'time' in request.GET:
                try:
                    return HttpResponse(
                        int((datetime.strptime(
                            request.GET['time'], settings.DATETIME_FORMAT,
                        ) - datetime.utcnow()).total_seconds())
                    )
                except ValueError:
                    # если не смогли распарсить дату, просто отдаем время
                    logger.warning(
                        'wrong `time` parameter in `/time/`, got: %s',
                        request.GET['time'])
            return HttpResponse(datetime.utcnow().strftime(
                settings.DATETIME_FORMAT))


class TVMMiddleware(MiddlewareMixin):
    """
    Проверяет TVM-тикет в запросе
    """
    service_ticket_headers = [
        'HTTP_X_YA_SERVICE_TICKET',
        'HTTP_X_TVM2_TICKET',
    ]

    user_ticket_header = 'HTTP_X_YA_USER_TICKET'

    @property
    def is_required(self):
        return getattr(settings, 'TVM_TICKET_REQUIRED', True)

    def get_service_ticket_headers(self):
        header = getattr(settings, 'TVM_SERVICE_TICKET_HEADER', None)
        if header:
            return [header]

        return self.service_ticket_headers

    def get_service_ticket(self, request):
        for header in self.get_service_ticket_headers():
            ticket = request.META.get(header)
            if ticket:
                return ticket

        return None

    def get_user_ticket(self, request):
        ticket_header = getattr(settings, 'TVM_USER_TICKET_HEADER', self.user_ticket_header)
        return request.META.get(ticket_header)

    def check_tickets(self, request):
        service_ticket = self.get_service_ticket(request)
        if service_ticket:
            # check service ticket
            try:
                parsed_service_ticket = tvm2_client.parse_service_ticket(service_ticket)
            except Exception as exc:
                logger.error(exc)
                return HttpResponseForbidden('Error parsing service ticket', status=401)
            if parsed_service_ticket:
                request.tvm_service_id = getattr(parsed_service_ticket, 'src', None)
            else:
                return HttpResponseForbidden('Unknown service ticket')

            # check user ticket
            user_ticket = self.get_user_ticket(request)
            if user_ticket:
                try:
                    parsed_user_ticket = tvm2_client.parse_user_ticket(user_ticket)
                except Exception as exc:
                    logger.error(exc)
                    return HttpResponseForbidden('Error parsing user ticket', status=401)
                if parsed_user_ticket:
                    request.tvm_uid = getattr(parsed_user_ticket, 'default_uid', None)
                else:
                    return HttpResponseForbidden('Unknown user ticket')

            return

        if self.is_required:
            return HttpResponseForbidden('TVM service ticket not provided.', status=401)

    def process_request(self, request):
        request.tvm_service_id = None
        request.tvm_uid = None

        normal_path = path.normpath(request.path) + '/'

        if any(map(normal_path.startswith, settings.TVM_EXCLUDE_PATHS)):
            return

        if settings.DEBUG and 'text/html' in request.META.get('HTTP_ACCEPT', ''):
            # Если сейчас DEBUG окружение, то из браузера можно попасть
            # куда угодно
            return

        return self.check_tickets(request)


class TVMDebugMiddleware(TVMMiddleware):
    debug_uid_header = 'HTTP_DEBUG_UID'
    debug_service_id_header = 'HTTP_DEBUG_SERVICE_ID'

    def get_debug_service_id(self, request):
        service_id = getattr(settings, 'DEBUG_TVM_SERVICE_ID', None)
        service_id = request.META.get(self.debug_service_id_header, service_id)
        return int(service_id) if service_id else None

    def get_debug_uid(self, request):
        debug_uid = getattr(settings, 'DEBUG_TVM_UID', None)
        debug_uid = request.META.get(self.debug_uid_header, debug_uid)
        return debug_uid

    def check_tickets(self, request):
        request.tvm_service_id = self.get_debug_service_id(request)
        request.tvm_uid = self.get_debug_uid(request)

        if request.tvm_service_id is None:
            return super().check_tickets(request)


class B2BAuthBackendMiddleware(MiddlewareMixin):
    """
        Данная миддлварь нужна только для режима изолированного б2б-тестирования API, когда
        у нас есть только username-ползователя и никаких сессионных кук.
        Этот режим предполагает наличие куки с юзернеймом яндекс-пользователя в запросе,
        по наличию этого юзернейма в нашей базе и происходит аутентификация.
        Эта миддлварь не предназначена для использования продакшена.
    """

    def process_request(self, request):
        request.yauser = authenticate(request=request)
        self.assign_user(request)  # для того, чтобы в request появился объект user

    def assign_user(self, request):
        if get_setting(['YAUSER_ADMIN_LOGIN', 'YAUTH_USE_NATIVE_USER']):
            request.__class__.user = DjangoUserDescriptor()


class FixEmptyHostMiddleware(MiddlewareMixin):
    """
    Used to fix bug with response code 400 when binding to [::] and empty
    Host header.
    """

    @staticmethod
    def process_request(request):
        if (
            'HTTP_HOST' not in request.META or
            request.META['HTTP_HOST'].startswith(':')
        ):
            request.META['HTTP_HOST'] = 'localhost'
