from builtins import object

from django.contrib.auth import get_user_model
from django.db import models
from django.utils.translation import ugettext_lazy as _

from .fields import JSONField

User = get_user_model()

INFO_SCHEMA = {
    'type': 'object',
    'properties': {
        'about': {
            'type': 'object',
            'properties': {
                'header': {
                    'type': 'string',
                },
                'text': {
                    'type': 'string',
                },
            },
            'additionalProperties': False,
            'required': ['header', 'text'],
        },
        'authors': {
            'type': 'array',
            'items': {
                'type': 'object',
                'properties': {
                    'photo': {
                        'type': 'string',
                    },
                    'fio': {
                        'type': 'string',
                    },
                    'about': {
                        'type': 'string',
                    },
                },
                'additionalProperties': False,
                'required': ['photo', 'fio', 'about'],
            },
        },
        'social': {
            'type': 'array',
            'items': {
                'type': 'object',
                'properties': {
                    'type': {
                        'type': 'string',
                    },
                    'link': {
                        'type': 'string',
                    },
                    'color': {
                        'type': 'string',
                    },
                    'text': {
                        'type': 'string',
                    },
                },
                'additionalProperties': False,
                'required': ['type', 'link'],
            },
        },
    },
    'additionalProperties': False,
}


class TimeStampMixin(models.Model):
    """
    Миксин добавляюищй в модель поля времени создания и обновления
    """
    date_created = models.DateTimeField(_('Creation date'), auto_now_add=True)
    date_updated = models.DateTimeField(_('Date of change'), auto_now=True)

    class Meta(object):
        abstract = True


class InfoMixin(models.Model):
    """
    Миксин добавляющий в модель поле информации
    """
    info = JSONField(
        verbose_name=_('Информация'),
        blank=True,
        null=True,
    )

    class Meta(object):
        abstract = True


class UserBlameMixin(models.Model):
    created_by = models.ForeignKey(
        to=User,
        verbose_name=_('Created by'),
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
        related_name='created_%(app_label)s_%(class)s'
    )
    modified_by = models.ForeignKey(
        to=User,
        verbose_name=_('Modified by'),
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
        related_name='modified_%(app_label)s_%(class)s'
    )

    class Meta(object):
        abstract = True


class AvailableForSupportMixin(models.Model):
    available_for_support = models.BooleanField(
        verbose_name=_('Доступен для саппорта'),
        default=False,
    )

    class Meta(object):
        abstract = True
