import logging
from builtins import object

import msgpack

from django.conf import settings

from kelvin.common.redis.redis_local_client import RedisLocalClient
from kelvin.common.redis.redis_sentinel_client import RedisSentinelClient
from kelvin.common.utils import generate_code

logger = logging.getLogger(__name__)


class RedisClient(object):
    REDIS_TYPES_MAP = {
        'sentinel': RedisSentinelClient,
        'local': RedisLocalClient,
        'default': RedisLocalClient,
    }

    def __init__(self, redis_type):
        self.redis_type = redis_type or 'default'

    def get_client(self):
        return self.REDIS_TYPES_MAP[self.redis_type]()


client = RedisClient(
    redis_type=settings.REDIS_CONNECTION_TYPE
).get_client()

ALPHABET = '0123456789ABCDEFGHIGKLMNOPQARSTUVWXYZabcdefghigklmnopqarstuvwxyz'
CODE_LENGTH = 6


def socketio_publish(nsp, room, data, event_name='message'):
    """
    Отправка сообщения в формате socket.io
    :param data: данные события
    :param nsp: неймспейс события
    :param room: идентификатор комнаты, в которой находятся пользователи
    :param event_name: название события
    """
    global client

    return client.publish(
        'socket.io#{0}#{1}#'.format(nsp, room),
        msgpack.packb([
            # случайный уникальный идентификатор события
            generate_code(CODE_LENGTH, alphabet=ALPHABET),
            {
                'type': 2,  # тип события — EVENT
                'data': [
                    event_name,
                    data,
                ],
                'nsp': nsp,
            },
            {
                'rooms': [room],
            },
        ]),
    )
