import logging

from redis.exceptions import BusyLoadingError
from rediscluster.client import StrictRedisCluster
from rediscluster.exceptions import ClusterDownError, ClusterError, RedisClusterException

from django.conf import settings

from kelvin.common.redis.redis_base_client import RedisBaseClient

logger = logging.getLogger(__name__)


class RedisClusterClient(RedisBaseClient):
    def __init__(self):
        self.__client = None

    def __create_redis_client(self):
        try:
            self.__client = StrictRedisCluster(
                startup_nodes=settings.REDIS_STARTUP_NODES,
                decode_responses=True,
            )
        except Exception as e:
            logger.exception(
                '[redis] Error when trying to create a connection: {}'
                .format(e)
            )

    def __safe_call(self, func):
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except (
                    RedisClusterException,
                    BusyLoadingError,
                    ClusterError,
                    ClusterDownError,
            ):
                logger.exception('[redis] Redis cluster unavailable')
            except KeyError:
                logger.exception('[redis] Redis cluster broken')
            except Exception as e:
                logger.exception('[redis] Exception: {}'.format(e))

            # in case of error remove saved client
            self.__client = None

        return wrapper

    def __getattr__(self, attr_name):
        if self.__client is None:
            logger.warning('[redis] Unavailable Redis client called')
            self.__create_redis_client()

            # If redis client was not created - return dummy
            if self.__client is None:
                logger.exception('[redis] Failed to create redis client')
                return lambda *_, **__: None

        # decorate __call__ for wrap it up in try/except
        attr = getattr(self.__client, attr_name)

        if not hasattr(attr, '__call__'):
            # attr not callable
            return attr

        return self.__safe_call(attr)
