import logging

from redis.exceptions import BusyLoadingError
from redis.sentinel import MasterNotFoundError, Sentinel

from django.conf import settings

from kelvin.common.redis.redis_base_client import RedisBaseClient

logger = logging.getLogger(__name__)


class RedisSentinelClient(RedisBaseClient):
    def __init__(self):
        self.__client = None
        self.sentinel_hosts = [
            (
                host,
                settings.REDIS_SENTINEL_PORT,
            )
            for host in settings.REDIS_SENTINEL_HOSTS
        ]

    def __create_redis_client(self):
        try:
            sentinel = Sentinel(
                self.sentinel_hosts,
                socket_timeout=settings.REDIS_SENTINEL_SOCKET_TIMEOUT
            )
            self.__client = sentinel.master_for(
                settings.REDIS_SENTINEL_MASTER,
                password=settings.REDIS_SENTINEL_PASSWORD
            )
        except Exception as e:
            logger.exception(
                '[redis] Error when trying to create a connection: {}'
                .format(e)
            )

    def __safe_call(self, func):
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except (
                    BusyLoadingError,
                    MasterNotFoundError,
            ):
                logger.exception('[redis] Redis sentinel unavailable')
            except KeyError:
                logger.exception('[redis] Redis sentinel broken')
            except Exception as e:
                logger.exception('[redis] Exception: {}'.format(e))

            # in case of error remove saved client
            self.__client = None

        return wrapper

    def __getattr__(self, attr_name):
        if self.__client is None:
            logger.warning('[redis] Unavailable Redis client called')
            self.__create_redis_client()

            # If redis client was not created - return dummy
            if self.__client is None:
                logger.exception('[redis] Failed to create redis client')
                return lambda *_, **__: None

        # decorate __call__ for wrap it up in try/except
        attr = getattr(self.__client, attr_name)

        if not hasattr(attr, '__call__'):
            # attr not callable
            return attr

        return self.__safe_call(attr)
