import logging
from builtins import object

import requests

from django.conf import settings

from kelvin.common.tvm import get_service_ticket

logger = logging.getLogger(__name__)


class RevisorError(Exception):
    pass


class Revisor(object):
    """
    Клиент для запросов в Revisor-микросервис
    """

    def __init__(self, url):
        self.url = url

    def request(self, url, method="get", data=None, params=None):
        """Отправка запроса в Reviewer"""
        resp = getattr(requests, method)(
            url,
            headers={
                "x-tvm2-ticket": get_service_ticket(settings.REVISOR_TVM_CLIENT_ID),
            },
            json=data or {},
            verify=False,
            params=params or {},
        )

        if resp.status_code != 200:
            raise RevisorError(
                'Unexpected status {}. Response text: {}'.format(
                    resp.status_code,
                    resp.text,
                )
            )

        if 'application/json' not in resp.headers['content-type']:
            raise RevisorError('Response is not json')

        data = resp.json()

        return data

    def create_user(self, **kwargs):
        response = self.request(
            url="{url}/users".format(url=self.url),
            data=kwargs,
            method="post"
        )
        logger.debug("Created revisor user: {}".format(response["id"]))
        return response

    def update_user(self, revisor_user_id, **kwargs):
        response = self.request(
            url="{url}/users/{user_id}".format(
                url=self.url,
                user_id=revisor_user_id,
            ),
            data=kwargs,
            method="put"
        )
        logger.debug("Updated revisor user: {}".format(revisor_user_id))
        return response

    def delete_user(self, revisor_user_id):
        response = self.request(
            url="{url}/users/{user_id}".format(
                url=self.url,
                user_id=revisor_user_id,
            ),
            method="delete"
        )
        logger.debug("Deleted revisor user: {}".format(revisor_user_id))
        return response

    def get_user(self, revisor_user_id):
        response = self.request(
            url="{url}/users/{user_id}".format(
                url=self.url,
                user_id=revisor_user_id,
            ),
            method="get"
        )
        logger.debug("Got revisor user: {}".format(revisor_user_id))
        return response

    def create_group(self, **kwargs):
        response = self.request(
            url="{url}/groups".format(url=self.url),
            data=kwargs,
            method="post"
        )
        logger.debug("Created revisor group: {}".format(response["id"]))
        return response

    def update_group(self, revisor_group_id, **kwargs):
        response = self.request(
            url="{url}/groups/{group_id}".format(
                url=self.url,
                group_id=revisor_group_id,
            ),
            data=kwargs,
            method="put"
        )
        logger.debug("Updated revisor group: {}".format(revisor_group_id))
        return response

    def delete_group(self, revisor_group_id):
        response = self.request(
            url="{url}/groups/{group_id}".format(
                url=self.url,
                group_id=revisor_group_id,
            ),
            method="delete"
        )
        logger.debug("Deleted revisor group: {}".format(revisor_group_id))
        return response

    def get_group(self, revisor_group_id):
        response = self.request(
            url="{url}/groups/{group_id}".format(
                url=self.url,
                group_id=revisor_group_id,
            ),
            method="get"
        )
        logger.debug("Got revisor group: {}".format(revisor_group_id))
        return response

    def create_group_membership(self, revisor_group_id, revisor_user_id, **kwargs):
        response = self.request(
            url="{url}/groups/{group_id}/users/{user_id}".format(
                url=self.url,
                group_id=revisor_group_id,
                user_id=revisor_user_id
            ),
            data=kwargs,
            method="post"
        )
        logger.debug("Added revisor user {} to revisor group".format(revisor_user_id, revisor_group_id))
        return response

    def update_group_membership(self, revisor_group_id, revisor_user_id, **kwargs):
        response = self.request(
            url="{url}/groups/{group_id}/users/{user_id}".format(
                url=self.url,
                group_id=revisor_group_id,
                user_id=revisor_user_id,
            ),
            data=kwargs,
            method="put"
        )
        logger.debug("Updated revisor user {} membership in group {}".format(revisor_user_id, revisor_group_id))
        return response

    def delete_group_membership(self, revisor_group_id, revisor_user_id):
        response = self.request(
            url="{url}/groups/{group_id}/users/{user_id}".format(
                url=self.url,
                group_id=revisor_group_id,
                user_id=revisor_user_id,
            ),
            method="delete"
        )
        logger.debug("Deleted revisor user {} from revisor group {}".format(revisor_user_id, revisor_group_id))
        return response

    def get_group_membership(self, revisor_group_id, revisor_user_id):
        response = self.request(
            url="{url}/groups/{group_id}/users/{user_id}".format(
                url=self.url,
                group_id=revisor_group_id,
                user_id=revisor_user_id,
            ),
            method="get"
        )
        logger.debug("Got revisor user {} membership in revisor group".format(revisor_user_id, revisor_group_id))
        return response

    def get_task(self, revisor_task_id):
        response = self.request(
            url="{url}/tasks/{task_id}".format(
                url=self.url,
                task_id=revisor_task_id,
            ),
            method="get"
        )
        logger.debug("Got revisor task: {}".format(revisor_task_id))
        return response

    def create_task(self, **kwargs):
        response = self.request(
            url="{url}/tasks".format(url=self.url),
            data=kwargs,
            method="post"
        )
        logger.debug("Created revisor task: {}".format(response["id"]))
        return response

    def update_task(self, revisor_task_id, **kwargs):
        response = self.request(
            url="{url}/tasks/{task_id}".format(url=self.url, task_id=revisor_task_id),
            data=kwargs,
            method="put"
        )
        logger.debug("Created revisor task: {}".format(response["id"]))
        return response

    def get_tasks(self, **kwargs):
        response = self.request(
            url="{url}/tasks".format(
                url=self.url,
            ),
            method="get",
            params=kwargs,
        )
        logger.debug("Got revisor task list")
        return response


def create_client():
    """Создание клиента для запросов в Revisor"""
    client = Revisor(
        settings.REVISOR_URL
    )
    return client


client = create_client()
