import logging
import os.path
from multiprocessing.pool import ThreadPool
from typing import Iterable

import boto3
from botocore.exceptions import ClientError

from django.conf import settings
from django.utils.functional import SimpleLazyObject

from .settings import S3_UNPACK_UPLOAD_THREADS

logger = logging.getLogger(__name__)


class S3BotoClient:
    def __init__(
        self,
        aws_access_key_id: str,
        aws_secret_access_key: str,
        endpoint_url: str,
        bucket: str,
    ):
        self.aws_access_key_id = aws_access_key_id
        self.aws_secret_access_key = aws_secret_access_key
        self.endpoint_url = endpoint_url
        self.bucket = bucket

        self.session = boto3.session.Session(
            aws_access_key_id=self.aws_access_key_id,
            aws_secret_access_key=self.aws_secret_access_key,
        )

        self.s3 = self.session.client(
            service_name='s3',
            endpoint_url=self.endpoint_url,
        )

    def file_exists(self, s3_file_path: str) -> bool:
        """
        Проверка существования файла на s3
        :param s3_file_path: путь на s3 к проверемому файлу
        """
        try:
            self.s3.head_object(Bucket=self.bucket, Key=s3_file_path)
        except ClientError as e:
            if int(e.response['Error']['Code']) == 404:
                return False
            raise

        return True

    def file_size(self, s3_file_path: str) -> str:
        try:
            head = self.s3.head_object(Bucket=self.bucket, Key=s3_file_path)
            return head['ContentLength']
        except ClientError as e:
            if int(e.response['Error']['Code']) == 404:
                raise Exception("File %s not found in s3" % s3_file_path)
            raise

    def download_file(self, s3_file_path: str, local_file_path: str) -> None:
        """
        Загружзка файла с s3
        :param s3_file_path: путь на s3 к загружаемому файлу
        :param local_file_path: локальный путь к загружаемому файлу
        """
        if not self.file_exists(s3_file_path):
            raise Exception("File %s not found in s3" % s3_file_path)

        self.s3.download_file(self.bucket, s3_file_path, local_file_path)

    def upload_file(self, local_file_path: str, s3_file_path: str) -> None:
        """
        Загрузка файла на s3
        :param local_file_path: локальный путь к загружаемому файлу
        :param s3_file_path: путь на s3 к загружаемому каталогу
        """
        with open(local_file_path, 'rb') as stream:
            self.s3.put_object(Bucket=self.bucket, Key=s3_file_path, Body=stream)

    def upload_folder(self, local_root_path: str, s3_root_path: str) -> None:
        """
        Рекурсивная загрузка каталога на s3
        :param local_root_dir: локальный путь к загружаемому каталогу
        :param s3_root_path: путь на s3 к загружаемому каталогу
        """
        # рекурсивный обход файлов в каталоге
        def walk_with_files(folder) -> Iterable[str]:
            for dirpath, dirnames, filenames in os.walk(folder):
                for filename in filenames:
                    yield os.path.join(dirpath, filename)  # полный путь к локальному файлу

        # загрузка файла на s3
        def upload_one_file(local_full_file_path) -> None:
            try:
                # относительный путь к локальному файлу
                file_path = os.path.relpath(path=local_full_file_path, start=local_root_path)
                self.upload_file(
                    local_file_path=local_full_file_path,
                    s3_file_path=os.path.join(s3_root_path, file_path),
                )
            except Exception as exc:
                logging.error(exc)
                raise

        pool = ThreadPool(processes=S3_UNPACK_UPLOAD_THREADS)
        pool.map(upload_one_file, walk_with_files(local_root_path))

    def delete_objects(self, s3_objects_list, quiet=True):
        parameters = {
            'Objects': [
                {'Key': key} for key in s3_objects_list
            ],
            'Quiet': quiet,
        }
        return self.s3.delete_objects(Bucket=self.bucket, Delete=parameters)

    def copy_object(self, s3_from_path, s3_to_path):
        source = f"{self.bucket}/{s3_from_path}"
        return self.s3.copy_object(CopySource=source, Bucket=self.bucket, Key=s3_to_path)

    def list(self, prefix, delimiter=''):
        return self.s3.list_objects(Bucket=self.bucket, Prefix=prefix, Delimiter=delimiter)


def get_client() -> S3BotoClient:
    return S3BotoClient(
        aws_access_key_id=settings.AWS_ACCESS_KEY_ID,
        aws_secret_access_key=settings.AWS_SECRET_ACCESS_KEY,
        endpoint_url=settings.AWS_S3_ENDPOINT_URL,
        bucket=settings.AWS_STORAGE_BUCKET_NAME,
    )


s3_boto_client = SimpleLazyObject(lambda: get_client())
