import base64
import json
import logging
import mimetypes
import os
from email.mime.base import MIMEBase

import requests
import six
from past.builtins import basestring
from requests.exceptions import ConnectionError, HTTPError, Timeout

from django.conf import settings
from django.core.mail import DEFAULT_ATTACHMENT_MIME_TYPE

from kelvin.accounts.models import User
from kelvin.celery import app
from kelvin.common.decorators.task_logger import logged_task

logger = logging.getLogger(__name__)


def send_message(
    receiver,
    template_name,
    sync=False,
    template_args=None,
    files=None,
    **kwargs
):
    """
    Use Sender API to send transaction letter (just through sender)

    Custom logic with receiver. It can be just str with email or User instance
    If it is user instance - try to send to Yandex Account email.

    :param receiver: email of receiver or User instance
    :param template_name: id of template in Sender
    :param sync: send letter right now (sync) or put it in queue (async)
    :param template_args: template args for use it inside template
    :param files: dict of files to attach
        key - filename, value - opened fileobject
    """
    if template_args is None:
        template_args = {}
    if files is None:
        files = {}

    data = {
        'args': json.dumps(template_args),
        'async': not sync,
    }
    attachments = []
    mimetypes.init()
    for filename, file_obj in list(files.items()):
        file_extension = os.path.splitext(filename)[1]
        try:
            mime_type = mimetypes.types_map.get(file_extension, '.pdf')
            # mimetype lib doesn't define csv file
            if mime_type is None and file_extension == '.csv':
                mime_type = 'text/csv'
            file_data = base64.b64encode(file_obj.read())
            attachments.append({
                'filename': filename,
                'mime_type': mime_type,
                'data': file_data,
            })
        except KeyError:
            logger.error(
                'Unknown mimetype for extension {}'.format(file_extension),
            )
    if attachments:
        data['attachments'] = json.dumps(attachments)

    # Get properly email
    email = ''
    if isinstance(receiver, basestring):
        email = receiver
    elif isinstance(receiver, User):
        email = receiver.email

    if not email:
        logger.info(
            'Empty email receiver. kwargs={kwargs}'
            .format(
                kwargs={
                    'receiver': receiver,
                    'template_name': template_name,
                    'async': not sync,
                    'template_args': template_args,
                    'files': files,
                    'user_id': getattr(receiver, 'id', None),
                },
            )
        )

        raise SenderError(
            'Try to send letter to empty email',
        )

    sender_response = sender_request(
        method='post',
        url='{api_url}/transactional/{template_name}/send'.format(
            api_url=settings.SENDER_API_URL,
            template_name=template_name,
        ),
        params={
            'to_email': email,
        },
        data=data,
        **kwargs
    )

    return sender_response


def sender_request(method, **kwargs):
    """
    Wrapper for sender special request
    Puts task to celery queue if task needs retry
    """
    try:
        return _make_sender_request(method, **kwargs)

    except SenderRetryException:
        # celery retry mailing task
        task_id = sender_request_task.delay(method, **kwargs)
        logger.warning('Sender celery retry called. task id %s', task_id)


@logged_task
@app.task(bind=True, acks_late=True, max_retry_count=2)
def sender_request_task(self, method, **kwargs):
    try:
        _make_sender_request(method, **kwargs)
    except SenderRetryException:
        logger.error('Sender celery retry')
        self.retry()


def _make_sender_request(method, **kwargs):
    """
    Special request for sender with auth, verify, retry and logging of response

    Raises SenderRetryException if none of MAX_RETRY_COUNT attempts succeeded
    and error needs retry
    """

    if 'auth' not in kwargs:
        kwargs['auth'] = (settings.SENDER_AUTHORIZATION_KEY, '')

    if 'verify' not in kwargs:
        kwargs['verify'] = settings.SENDER_VERIFY_HOST

    # Set default timeout for request to Sender
    if 'timeout' not in kwargs:
        kwargs['timeout'] = settings.SENDER_REQUEST_TIMEOUT

    kwargs['method'] = method

    attempts = 0

    while attempts < settings.SENDER_MAX_RETRY_COUNT:
        try:
            attempts += 1
            sender_response = requests.request(**kwargs)

        except ConnectionError as e:
            # Max retries exceeded with url - retry
            logger.info('Sender ConnectionError %s', e)
            continue

        except Timeout as e:
            # Read timed out - not safe to retry, retry only important mails
            if 'url' in kwargs and any(
                t in kwargs['url']
                for t in settings.SENDER_RETRY_TEMPLATES
            ):
                continue
            last_exception = e
            logger.info('Sender Timeout %s', e)
            break
        except BaseException as e:
            # unhandled exception - not safe to retry
            last_exception = e
            logger.error(
                'Unhandled Sender request error: {error}\n Request params: '
                '{params}'.format(
                    error=e.message,
                    params=kwargs,
                )
            )
            break

        try:
            parse_sender_response(sender_response)
            return sender_response.json()

        except SenderRetryException:
            continue
    else:
        # if not break
        raise SenderRetryException()

    raise SenderError(
        last_exception.message if last_exception else 'An error occurred'
    )


def parse_sender_response(sender_response):
    """
    Parses response to check whether retry is needed

    returns sender_response if Ok or no retry needed
    raises SenderRetryException if request needs retry
    """
    if sender_response.status_code == 200:
        return sender_response

    sender_error_text = (
        'Error while using Sender\n'
        '\tStatus code: {code}\n'
        '\tReason: {reason}\n'
        '\tHeaders: {headers}\n'
        '\tText: {text}\n'
        '\tURL: {url}'.format(
            code=sender_response.status_code,
            reason=sender_response.reason,
            headers=sender_response.headers,
            text=sender_response.text,
            url=sender_response.url,
        )
    )

    if 400 <= sender_response.status_code < 500:
        # incorrect request data --> no retry

        # invalid or not existing email -> no warning
        # {"result":{"status":"ERROR","error":{"to_email":["Invalid value"]}}}
        try:
            text = json.loads(sender_response.text)
        except ValueError:
            logger.warning(sender_error_text)
            return sender_response
        try:
            if text['result']['error']['to_email'] != ['Invalid value']:
                logger.warning(sender_error_text)
        except KeyError:
            logger.warning(sender_error_text)

        return sender_response

    # Internal Sender service error --> retry
    logger.warning(sender_error_text)
    raise SenderRetryException(sender_response=sender_response)


class SenderRetryException(Exception):

    def __init__(self, sender_response=None):
        self.sender_response = sender_response


class SenderError(Exception):
    pass


class SenderHTTPError(HTTPError):
    pass


class SenderEmail:
    template_name = None

    def __init__(self, to_email=None, variables=None, attachments=None, sync=False, **kwargs):
        self.to = to_email
        self.variables = variables
        self.sync = sync

        self.attachments = []
        if attachments:
            for filename, content in list(attachments.items()):
                self.attach(filename, content)

        if 'auth' not in kwargs:
            kwargs['auth'] = (settings.SENDER_AUTHORIZATION_KEY, '')

        if 'verify' not in kwargs:
            kwargs['verify'] = settings.SENDER_VERIFY_HOST

        if 'timeout' not in kwargs:
            kwargs['timeout'] = settings.SENDER_REQUEST_TIMEOUT

        self.kwargs = kwargs

    def attach(self, filename, content, mimetype=None):
        if not mimetype:
            mimetype, _ = mimetypes.guess_type(filename)
            if not mimetype:
                mimetype = DEFAULT_ATTACHMENT_MIME_TYPE
        basetype, subtype = mimetype.split('/', 1)

        if basetype == 'text':
            if isinstance(content, six.binary_type):
                try:
                    content = content.decode('utf-8')
                except UnicodeDecodeError:
                    mimetype = DEFAULT_ATTACHMENT_MIME_TYPE

        self.attachments.append({
            'filename': filename,
            'mime_type': mimetype,
            'data': base64.b64encode(content.read())
        })

    @property
    def recipient(self) -> str:
        recipient = self.to

        if isinstance(recipient, User):
            recipient = recipient.email

        return recipient

    @property
    def message(self) -> dict:
        payload = {
            'to_email': self.recipient,
            'args': json.dumps(self.variables or {}),
            'async': not self.sync,
        }

        if self.attachments:
            payload['attachments'] = json.dumps(self.attachments)

        return payload

    @property
    def template(self) -> str:
        assert self.template_name is not None, (
            "Property 'template_name' should be set for '%s'" % self.__class__.__name__
        )

        template_id = settings.SENDER_TEMPLATES.get(self.template_name)
        assert template_id is not None, (
            "Template '%s' is not found" % self.template_name
        )

        return template_id

    def _request(self, method='post', **extra_kwargs):
        url = '{}/transactional/{}/send'.format(
            settings.SENDER_API_URL,
            self.template,
        )

        kwargs = self.kwargs
        kwargs.update(extra_kwargs)
        return self._response(requests.request(method, url, **kwargs))

    def _response(self, response):
        if response.status_code == 200:
            return response

        error_message = (
            'Error while using Sender\n'
            '\tStatus code: {code}\n'
            '\tReason: {reason}\n'
            '\tHeaders: {headers}\n'
            '\tText: {text}\n'
            '\tURL: {url}'.format(
                code=response.status_code,
                reason=response.reason,
                headers=response.headers,
                text=response.text,
                url=response.url,
            )
        )

        if 400 <= response.status_code < 500:
            try:
                text = json.loads(response.text)
            except ValueError:
                logger.warning(error_message)
                return response

            return response

        logger.warning(error_message)
        raise SenderHTTPError(response=response)

    def send(self, **kwargs):
        if not self.recipient:
            return

        response = self._request(
            data=self.message,
            **kwargs,
        )
        return response.json()
