import datetime
import time
from builtins import map, object
from collections import Mapping

import jsonschema

from rest_framework import fields, serializers

from kelvin.common.utils import dt_from_microseconds, dt_to_microseconds


class SelectRelatedField(object):
    """
    Служит для оптимизации количества запросов при получении данных о
    вложенных ForeignKey-полях, пример использования есть в документации
    к PrefetchRelationListSerializer.
    """

    def __init__(self, name):
        """
        :param name: имя поля, которое нужно подставить в select_related
            запроса.
        """
        self.name = name


class PrefetchRelatedField(object):
    """
    Служит для оптимизации количества запросов при сохранении
    вложенных ForeignKey-полей, пример использования есть в документации
    к PrefetchRelationListSerializer.
    """
    context_key_template = 'prefetch__{name}'

    def __init__(self, name, queryset, pk_name='id'):
        """
        :param name: имя поля в сериализаторе, содержимое которого
            класс-наследник NestedForeignKeyField.
        :param queryset: кверисет, который будет использован для
            прекэширования данных.
        :type queryset: django.db.models.query.QuerySet
        :param pk_name: имя поля-первичного ключа в данных, которые приходят с
            фронтэнда, например если с фронта приходит словарик вида
            `{"id": 1212, "body": "Some content"}`, то pk_name = 'id'.
        """
        self.name = name
        self.pk_name = pk_name
        self.queryset = queryset

    def _extract_id(self, item):
        """
        Достает из объекта его идентификатор, возможные варианты:

        1. {name: <id>}
        2. {name: {pk_name: <id>}}
        """
        if not isinstance(item, Mapping):
            return None

        instance = item.get(self.name)

        if not instance:
            return None

        if isinstance(instance, Mapping):
            # {"id": some integer }
            return instance.get(self.pk_name)
        elif isinstance(instance, int):
            # просто id
            return instance

        return None

    def get_instance_mapping(self, data):
        """
        Возвращает словарь с закэшированными данными, вида:
        `{id_1: instance_1, id_2: instance_2, ...}`

        :type data: list[dict]
        """
        # получаем идентификаторы объектов, соответствующих полю self.field
        # отфильтруем None из списка
        ids = [_f for _f in map(self._extract_id, data) if _f]

        if not ids:
            return {}

        return self.queryset.in_bulk(ids)

    @property
    def prefetch_key(self):
        return self.get_prefetch_key(self.name)

    @classmethod
    def get_prefetch_key(cls, field_name):
        """
        Название ключа в словарике context у сериализатора.
        """
        return cls.context_key_template.format(name=field_name)


class NestedRelationalFieldBase(serializers.Field):
    """
    Базовый класс для полей, реализующих отношения
    """
    class Meta(object):
        serializer = None
        model = None

    def __init__(self, *args, **kwargs):
        super(NestedRelationalFieldBase, self).__init__(*args, **kwargs)
        assert hasattr(self.Meta, 'model') and self.Meta.model is not None, (
            'Model must be specified in Meta')
        assert (hasattr(self.Meta, 'serializer') and self.Meta.serializer is not None), (
            'Serializer class must be specified in Meta')


class NestedForeignKeyMixin(object):
    """
    Позволяет использовать 2 формата указания идентификатора объекта
    в запросе на запись\изменения данных:

    1. {
        'object': 666  # object id
    }

    2. {
        'object': {'id': 666}  # object id
    }
    """

    default_error_messages = {
        'bad value': 'bad value in {field} data: {item}',
        'doesnt exist': (
            'Some objects for {field} doesn\'t exist: {id}'),
        'id missing': (
            'ID attribute wasn\'t found for {field} in {item}'),
    }

    def _get_value(self, instance_id):
        """
        Прежде чем сделать запрос в БД за объектом, пытаем поискать его
        закэшированную версию в `context`, см. PrefetchRelationListSerializer

        :type self: rest_framework.serializers.Field
        """
        for attr in getattr(self, 'source_attrs', []):
            prefetched = self.context.get(
                PrefetchRelatedField.get_prefetch_key(attr)
            )

            if not prefetched:
                break

            if prefetched and instance_id in prefetched:
                return prefetched[instance_id]

        return self.Meta.model.objects.get(id=instance_id)

    def to_internal_value(self, data):
        """
        Десериализует из значений двух видов:
        1) словарь, в котором указан идентификатор
        2) идентификатор

        :type self: NestedForeignKeyMixin |
            rest_framework.serializers.ModelSerializer
        """
        if isinstance(data, dict):
            if 'id' not in data:
                self.fail('id missing',
                          field=self.label or self.field_name,
                          item=data)
            instance_id = data['id']
        else:
            instance_id = data

        try:
            internal_value = self._get_value(instance_id)
        except (TypeError, ValueError):
            self.fail('bad value',
                      field=self.label or self.field_name,
                      item=data)
        except self.Meta.model.DoesNotExist:
            self.fail('doesnt exist',
                      field=self.label or self.field_name,
                      id=instance_id)
        return internal_value


class NestedForeignKeyField(NestedForeignKeyMixin, NestedRelationalFieldBase):
    """
    Поле, в которое можно записывать идентификаторы и которое сериализует
    foreign key модель
    """

    def to_representation(self, value):
        """
        Возвращает сериализованные объекты с помощью сериализатора,
        указанного в `Meta` классе
        """
        return (self.Meta.serializer(context=self.context)
                .to_representation(value))


class NestedManyToManyField(NestedRelationalFieldBase):
    """
    Поле для наследования для реализации отношения `ManyToMany`
    Сохранение реализуется при помощи списка `id` или списка словарей с
    ключем `id`
    """
    default_error_messages = {
        'not list': '{field} field requires list value',
        'bad value': 'bad value in {field} data: {item}',
        'doesnt exist': (
            'Some objects for {field} doesn\'t exist: {ids}'),
        'id missing': (
            'ID attribute wasn\'t found for {field} in {item}'),
    }

    def to_internal_value(self, data):
        """
        Десериализует из значений двух видов:
        1) список словарей, в каждом из которых указан идентификатор
        2) список идентификаторов
        """
        if not isinstance(data, list):
            self.fail('not list', field=self.label or self.field_name)
        internal_value = []
        for item in data:
            try:
                if isinstance(item, dict):
                    if 'id' not in item:
                        self.fail('id missing',
                                  field=self.label or self.field_name,
                                  item=item)
                    internal_value.append(int(item['id']))
                else:
                    internal_value.append(int(item))
            except (KeyError, TypeError, ValueError):
                self.fail('bad value',
                          field=self.label or self.field_name,
                          item=item)
        existed_values = set(self.Meta.model.objects.filter(
            id__in=internal_value).values_list('id', flat=True))
        if existed_values != set(internal_value):
            self.fail('doesnt exist',
                      field=self.label or self.field_name,
                      ids=set(internal_value) - existed_values)
        return internal_value

    def to_representation(self, value):
        """
        Возвращает сериализованные объекты с помощью сериализатора,
        указанного в `Meta` классе
        """
        return (self.Meta.serializer(many=True, context=self.context)
                .to_representation(value))


class UnixTimeField(fields.Field):
    """
    Поле, представляющее дату-время как число unixtime.
    """

    def to_representation(self, value):
        """
        Представляет дату и время в виде unixtime
        """
        return int(time.mktime(value.timetuple()))

    def to_internal_value(self, data):
        """
        Делает `datetime` из числа unixtime
        """
        try:
            return datetime.datetime.fromtimestamp(int(data))
        except (TypeError, ValueError):
            raise serializers.ValidationError("Incorrect unixtime: {0}"
                                              .format(data))


class MicrosecondsDateTimeField(fields.Field):
    """
    Поле, представляющее дату-время как число микросекунд
    """

    def to_representation(self, value):
        """
        Вернуть число микросекунд в дате
        """
        return dt_to_microseconds(value)

    def to_internal_value(self, data):
        """
        Делает `datetime` из числа микросекунд
        """
        try:
            return dt_from_microseconds(data)
        except (TypeError, ValueError):
            raise serializers.ValidationError(
                'Incorrect microseconds: {0}'.format(data))


class JSONField(serializers.Field):
    """
    Сериализация и десериализация поля разметки задачи
    """
    # json схема для валидации значения поля
    json_schema = None

    def __init__(self, *args, **kwargs):
        """
        Инициализирует `json_schema`. Также по умолчанию принимает null
        как валидный json
        """
        self.json_schema = kwargs.pop('schema', None) or self.json_schema
        kwargs.setdefault('allow_null', True)
        super(JSONField, self).__init__(*args, **kwargs)

    def to_representation(self, value):
        """
        Сериализация поля разметки задачи
        """
        return value

    def to_internal_value(self, data):
        """
        Десериализация поля разметки задачи
        Валидирует значение поля согласно схеме, указанной при инициализации
        :raise serializers.ValidationError: если значение не удовлетворяет
        схеме
        """
        if self.json_schema is not None:
            try:
                jsonschema.validate(data, self.json_schema)
            except jsonschema.ValidationError as e:
                raise serializers.ValidationError(e.message)
        return data


class JSONDictField(JSONField):
    """
    JSON поле, позволяющее сохранять только словарь
    """
    json_schema = {'type': 'object'}


class CurrentUserOrNoneDefault(fields.CurrentUserDefault):
    """
    Значение по умолчанию для пользователя, авторизованного или нет.
    """

    def __call__(self):
        if self.user.is_authenticated:
            return self.user
        return None
