from builtins import object
from datetime import datetime

from django.db import models, transaction
from django.utils import timezone

from rest_framework.serializers import ModelSerializer
from rest_framework.utils import model_meta

from kelvin.common.model_mixins import INFO_SCHEMA
from kelvin.common.serializer_fields import JSONField, MicrosecondsDateTimeField, UnixTimeField


class SetFieldsMixin(object):
    """
    Позволяет передавать в сериализатор, какие поля нужно отобразить

    Используется с `rest_framework.serializers.ModelSerializer`
    """

    def __init__(self, *args, **kwargs):
        """
        Даем возможность указать поля, которые нужны для отображения
        """
        # http://stackoverflow.com/questions/18696403/dynamically-modifying-serializer-fields-in-django-rest-framework
        # Don't pass the 'fields' arg up to the superclass
        fields = kwargs.pop('fields', None)

        # Instantiate the superclass normally
        super(SetFieldsMixin, self).__init__(*args, **kwargs)

        if fields:
            # Drop any fields that are not specified in the `fields` argument.
            allowed = set(fields)
            existing = set(self.fields.keys())
            for field_name in existing - allowed:
                self.fields.pop(field_name)


class SkipNullsMixin(object):
    """
    Миксин для отбрасывания пустых ключей в сериализованных данных
    """

    def to_representation(self, instance):
        """
        Убирает из сериализованных данных поля со значением `None`
        """
        data = super(SkipNullsMixin, self).to_representation(instance)
        data = {
            key: value
            for key, value in data.items() if value is not None
        }
        return data


class DateFieldsYearLimitMixin(object):
    """
    Mixin for fields of DateTimeField and DateField
    if date is not in unix time, replaces it with 1970 year
    for escape errors of invalid dates
    """

    def to_representation(self, instance):
        if isinstance(instance, models.Model):
            for field in type(instance)._meta.fields:
                if isinstance(field, (models.DateTimeField, models.DateField)):
                    value = getattr(instance, field.name)
                    if value and value.year < 1970:
                        _1970 = datetime.fromtimestamp(1, tz=timezone.utc)
                        if isinstance(field, models.DateTimeField):
                            setattr(instance, field.name, _1970)
                        else:
                            setattr(instance, field.name, _1970.date())

        return (
            super(DateFieldsYearLimitMixin, self).to_representation(instance)
        )


class DateUpdatedFieldMixin(object):
    """
    Миксин к классу `ModelSerializer` для представления времени обновления в
    виде unixtime
    """

    def build_field(self, field_name, info, model, depth):
        """
        Подменим класс поля `date_updated` на `UnixTimeField`
        """
        field_class, field_kwargs = super(
            DateUpdatedFieldMixin, self).build_field(field_name, info,
                                                     model, depth)
        if field_name == 'date_updated':
            field_class = UnixTimeField

        return field_class, field_kwargs


class PreciseDateUpdatedFieldMixin(object):
    """
    Миксин к классу `ModelSerializer` для представления времени обновления в
    виде unixtime
    """

    def build_field(self, field_name, info, model, depth):
        """
        Подменим класс поля `date_updated` на `UnixTimeField`
        """
        field_class, field_kwargs = (
            super(PreciseDateUpdatedFieldMixin, self)
            .build_field(field_name, info, model, depth)
        )
        if field_name == 'date_updated':
            field_class = MicrosecondsDateTimeField

        return field_class, field_kwargs


class InfoFieldMixin(object):
    info = JSONField(schema=INFO_SCHEMA, allow_null=True)


class SerializerManyToManyMixin(object):
    """
    Миксин для сериализации и десериализации модели с полями `ManyToMany`
    Для использования необходимо переопределение
    поля `m2m_update_fields`, которое задается в формате:
    {field_name: related_field_name}
    где field_name - имя поля many_to_many,
                     должно совпадать с именем поля в моделе
        related_field_name - имя поля, соотв. данному сериализатору

    Пример:

    class Author(models.Model):
        name = models.CharField()
        songs = models.ManyToMany(Author, through='AuthorSong')

    class AuthorSong(models.Model):
        song = models.ForeignKey(Song)
        author = models.ForeignKey(Author)
        year = models.IntegerField()

    class Song(models.Model):
        name = models.CharField()

    сериализатор:

    class AuthorSerializer(SerializerManyToMany, ModelSerializer):
        m2m_update_fields = {'songs': 'author'}
        songs = SongSerializer(source='authorsong_set', many=True)
    """
    m2m_update_fields = None

    # Список полей, для которых не нужно обновлять many-to-many связь,
    # например если используется ManyToManyListSerializer, который все
    # нужные связи создает при вызове метода save().
    #
    # Указание здесь поля поможет экономить UPDATE/INSERT запросы.
    m2m_skip_update = ()

    def __init__(self, *args, **kwargs):
        """
        :type self: SerializerManyToManyMixin | ModelSerializer
        """
        super(SerializerManyToManyMixin, self).__init__(*args, **kwargs)
        assert self.m2m_update_fields is not None, (
            'Field m2m_update_fields is not declared')

    def save(self, **kwargs):
        """
        Сохранение сериализатора. Сначала сохраняет саму модель, а потом все
        поля, указанные в `m2m_update_fields`

        :type self: SerializerManyToManyMixin | ModelSerializer
        """
        with transaction.atomic():
            # убираем значения, модели для которых создаем сами
            m2m_validated_data = {}
            for field, related_field in list(self.m2m_update_fields.items()):
                # в validated_data данные находятся по ключу, указанному как
                # source в параметрах поля
                field_name = self.fields[field].source
                if field_name in self.validated_data:
                    m2m_validated_data[field] = self.validated_data.pop(
                        field_name)

            # сперва сохраняем саму модель
            instance = super(SerializerManyToManyMixin, self).save(**kwargs)

            # сохраняем m2m-сериализаторы
            for field, related_field in list(self.m2m_update_fields.items()):
                m2m_serializer = self.fields[field]
                if field not in m2m_validated_data:
                    # для поля не переданы данные, частичное изменение
                    continue
                m2m_serializer._validated_data = m2m_validated_data[field]
                field_name = self.fields[field].source

                if self.instance:
                    instances = getattr(self.instance, field_name)
                    if instances:
                        m2m_serializer.instance = instances.all()

                m2m_results = m2m_serializer.save(**{related_field: instance})
                if field not in self.m2m_skip_update:
                    setattr(instance, field_name, m2m_results)

        return instance


class SerializerForeignKeyMixin(object):
    """
    Миксин для `serializers.ModelSerializer`. Дает возможность создания
    объектов моделей во вложенных сериализаторах. Для этого нужно в
    поле `fk_update_fields` добавить список с названиями тех полей,
    для которых нужна возможность создания.
    Для простого указания связи в поле должен лежать id объекта.

    Пример:

    class Author(models.Model):
        name = models.CharField()
        group = models.ForeignKey(Group)


    class Group(models.Model):
        name = models.CharField()
        creation_year = models.IntegerField()


    class AuthorSerializer(SerializerForeignKeyMixin,
                           serializers.ModelSerializer):
        fk_update_fields = ['group']
        group = GroupSerializer()
    """
    fk_update_fields = []

    def __init__(self, *args, **kwargs):
        """
        Проверки правильного использования миксина
        """
        super(SerializerForeignKeyMixin, self).__init__(*args, **kwargs)
        assert isinstance(self, ModelSerializer), (
            u"SerializerForeignKeyMixin must be used with a "
            u"class inheriting from `ModelSerializer`"
        )
        assert self.fk_update_fields, (
            u"`fk_update_fields` was not set in `{0}`"
            .format(self.__class__.__name__)
        )

    def get_fields(self):
        """
        Обновление поля `partial` у дочернего сериализатора (делаем здесь,
        чтобы при инициализации не обращаться к полям сериализатора)
        """
        fields = super(SerializerForeignKeyMixin, self).get_fields()
        for field_name in self.fk_update_fields:
            if field_name not in fields:
                continue
            assert isinstance(fields[field_name], ModelSerializer), (
                u"Cannot use `ForeignKeyMixin` with field '{0}', which is not "
                u"an instance of `ModelSerializer`".format(field_name)
            )
            fields[field_name].partial = self.partial
        return fields

    def to_internal_value(self, data):
        """
        Если поле указано в `fk_update_fields` и в данных по нему лежит
        не словарь, то заменяем сериализатор в этом поле на
        `PrimaryKeyRelatedField`, которое инициализруем с
        помощью `ModelSerializer.build_relational_field`
        """
        if isinstance(data, dict):
            info = model_meta.get_field_info(self.Meta.model)
            for fk_field_name in self.fk_update_fields:
                if fk_field_name not in self.fields:
                    continue
                if fk_field_name in data and not isinstance(data[fk_field_name], dict):
                    relation_info = info.relations[fk_field_name]
                    field_class, field_kwargs = (
                        self.build_relational_field(fk_field_name,
                                                    relation_info))
                    field_kwargs.update(self.fields[fk_field_name]._kwargs)
                    self.fields[fk_field_name] = field_class(**field_kwargs)
        return super(SerializerForeignKeyMixin, self).to_internal_value(data)

    def save(self, **kwargs):
        """
        Сохранение сериализатора. Сначала создает объекты требуемых вложенных
        сериализаторов, затем объект модели родительского сериализатора. Все
        выполняется в пределах транзакции. Учитывает возможность того, что в
        валидированных данных поля может быть инстанс модели
        """
        with transaction.atomic():
            for field_name in self.fk_update_fields:
                if field_name not in self.fields:
                    continue
                if field_name not in self.validated_data:
                    continue
                field_data = self.validated_data[field_name]
                nested = self.fields[field_name]
                if isinstance(field_data, models.Model):
                    continue
                elif isinstance(field_data, dict):
                    if self.instance:
                        # изменение родительского объекта
                        nested_instance = getattr(self.instance, nested.source)
                        if ('id' in field_data and (
                            not nested_instance or (
                                nested_instance.id != field_data['id']))):
                            # случай замены возможно существующей связи и
                            # изменение нового привязываемого объекта
                            nested.instance = (nested.Meta.model.objects
                                               .get(id=field_data['id']))
                        else:
                            # случай обновления существующего объекта в связи
                            # или создания нового объекта для связи
                            nested.instance = nested_instance
                    elif 'id' in field_data:
                        # создание родительского объекта с существующим
                        # объектом для связи
                        nested.instance = (nested.Meta.model.objects
                                           .get(id=field_data['id']))
                nested._validated_data = field_data
                nested._errors = {}
                self.validated_data[field_name] = nested.save()

            return super(SerializerForeignKeyMixin, self).save(**kwargs)


class ExcludeForStudentMixin(object):
    """
    Если в контексте сериализатора указан `for_student=True`, фильтрует
    поля сериализатора, исключая поля, указанные в `Meta.exclude_for_student`
    """

    def get_field_names(self, declared_fields, info):
        field_names = super(ExcludeForStudentMixin, self).get_field_names(
            declared_fields, info)
        if self.context.get('for_student'):
            field_names = [
                name for name in field_names
                if name not in self.Meta.exclude_for_student
            ]
        return field_names
