import datetime
from collections import OrderedDict

from django.conf import settings
from django.core.serializers.json import DjangoJSONEncoder
from django.db import models, transaction
from django.utils.functional import cached_property

from rest_framework import serializers

from kelvin.common.serializer_fields import PrefetchRelatedField, SelectRelatedField


class DictSerializer(serializers.ListSerializer):
    """
    Сериализует список объектов в вид
    {
        obj1.pk: {**все_сериализованные_поля_объкекта},
        obj2.pk: ...
    }
    """

    def to_representation(self, data):
        """
        Сериализуем каждый объект сериализатором `self.child`,
        кладём его в словарь по ключу `obj.pk`
        """
        iterable = data.all() if isinstance(
            data, (models.Manager, models.query.QuerySet)) else data

        return OrderedDict(
            (obj.pk, self.child.to_representation(obj)) for obj in iterable
        )


class PrefetchRelationListSerializer(serializers.ListSerializer):
    """
    Позволяет сократить запросы к БД для сериализаторов, которые используют
    ManyToManyListSerializer или NestedForeignKeyMixin.

    "Подтягивает" данные из БД для указанных FK-полей одним запросом, либо
    добавляет необходимые `select_related` к запросам из метода
    `to_representation` сериализаторов.

    Пример использования:

    >>> # В данной конфигурации мы будем получать N+1 запросов на
    >>> # получение FK-поля `problem` при получении и сохранении данных
    class ProblemNestedField(NestedForeignKeyField):
        class Meta:
            serializer = ProblemSerializer
            model = Problem

    class LessonProblemLinkSerializer(serializers.ModelSerializer):
        id = serializers.IntegerField(required=False)
        problem = ProblemNestedField(allow_null=True, required=False)

        class Meta:
            model = LessonProblemLink
            ...
            list_serializer_class = ManyToManyListSerializer

    >>> LessonProblemLinkSerializer(queryset, many=True).data
    ... # N+1 запросов SELECT
    >>> LessonProblemLinkSerializer(data=data, many=True).save()
    ... # N+1 запросов SELECT

    Добавляем prefetching, чтобы сократить количество запросов до 2-х:

    class LessonProblemLinkSerializer(serializers.ModelSerializer):
        ...
        prefetch_fk_fields = (
            # прекэшируем данные по id в `data`, чтобы сократить
            # количество запросов при сохранении
            PrefetchRelatedField('problem', Problem.objects.all()),
            # делаем join, чтобы сразу запросить из базы
            # LessonProblemLink и Problem
            SelectRelatedField('problem'),
        )
        ...
    """
    class_attr_field = 'prefetch_fk_fields'

    @cached_property
    def _prefetch_fields(self):
        """
        Возращает список объектов типа PrefetchRelatedField и
        SelectRelatedField если они объявлены у сериализатора.
        """
        return getattr(self.child, self.class_attr_field, [])

    def to_representation(self, data):
        if self._prefetch_fields:
            data = self._prefetch_data_to_representation(data)

        return (
            super(PrefetchRelationListSerializer, self).to_representation(data)
        )

    def to_internal_value(self, data):
        if self._prefetch_fields:
            self._prefetch_data_to_internal_value(data)

        return (
            super(PrefetchRelationListSerializer, self).to_internal_value(data)
        )

    def _prefetch_data_to_representation(self, data):
        """
        В случае если DRF обрабатывает ListSerializer, подправляем
        полученный QuerySet или Manager, чтобы при запросе он выполнил
        select_related.
        """
        if not isinstance(data, (models.Manager, models.QuerySet)):
            return data

        select_related_fields = [
            field.name for field in self._prefetch_fields
            if isinstance(field, SelectRelatedField)
        ]

        return data.select_related(*select_related_fields)

    def _prefetch_data_to_internal_value(self, data):
        """
        Прекеширует данные, для дальнейшенго использования их в
        NestedForeignKeyMixin: получаем все объекты, которые могут
        понадобиться по их id и помещаем в `context` сериализатора.
        """
        assert isinstance(data, list), (
            'PrefetchRelationMixin must be used as ListSerializer only'
        )

        for prefetch_object in self._prefetch_fields:
            if not isinstance(prefetch_object, PrefetchRelatedField):
                continue

            prefetch_mapping = prefetch_object.get_instance_mapping(data)

            if prefetch_mapping:
                self.context[prefetch_object.prefetch_key] = prefetch_mapping


class ManyToManyListSerializer(PrefetchRelationListSerializer):
    """
    Вспомогательный сериализатор для редактирования полей ManyToMany с through
    параметром
    Сохраняет существующие связи по уникальности, указанной в unique_together
    модели
    """

    @transaction.atomic()
    def update(self, instances, validated_data):
        """
        Обновление инстансов связей

        :param instances: список существующих связей

        TODO: этот метод генерирует к базе данных количество запросов как
        минимум равное количеству элементов, которые нужно обновить\создать.

        Оптимизация этого поведения достаточно трудоемкий процесс, т.к.
        DRF делает в методах update и create необходимые обработки, которые
        необходимо будет оттуда перенести + выполняются сигналы из ORM.

        http://www.django-rest-framework.org/api-guide/serializers/#customizing-multiple-create  # noqa
        http://www.django-rest-framework.org/api-guide/serializers/#customizing-multiple-update  # noqa
        """
        update_data = {}
        create_data = []
        for item in validated_data:
            if 'id' in item:
                update_data[item['id']] = item
            else:
                create_data.append(item)

        instance_mapping = {instance.id: instance for instance in instances}

        # Удаляем ненужные связи
        # N запросов на DELETE
        delete_ids = set(instance_mapping) - set(update_data)
        for instance_id in delete_ids:
            instance = instance_mapping[instance_id]
            instance.delete()

        # Обновляем связи
        # N запросов на UPDATE
        ret = []
        for instance_id, data in update_data.items():
            instance = instance_mapping.get(instance_id, None)

            # кто-то уже удалил сущность объекта с таким id, ну ок
            if not instance:
                continue
            ret.append(self.child.update(instance, data))

        # Создаем связи
        # N запросов на INSERT
        for data in create_data:
            ret.append(self.child.create(data))

        return ret


class CustomDjangoJSONEncoder(DjangoJSONEncoder):
    """
    Подкласс DjangoJSONEncoder, который кодирует datetime.datetime форматом,
    взятым из настроек кельвина.
    """

    def default(self, object):
        if isinstance(object, datetime.datetime):
            result = object.strftime(settings.DATETIME_FORMAT)
            return result
        else:
            return super(CustomDjangoJSONEncoder, self).default(object)
