import logging
import re
import sys
from builtins import object, range, str
from typing import Optional

import requests
import urllib3
from requests import ConnectionError
from requests.compat import urljoin
from urllib3.exceptions import InsecureRequestWarning

from django.conf import settings
from django.contrib.auth.models import Group
from django.core.cache import caches

logger = logging.getLogger(__name__)
urllib3.disable_warnings(category=InsecureRequestWarning)
main_cache = caches['main_cache']
staff_reader_cache = caches['staff_reader']


def chunks(l, n):
    """Yield successive n-sized chunks from l."""
    for i in range(0, len(l), n):
        yield l[i:i + n]


class StaffReaderError(Exception):
    pass


class StaffReader(object):
    """Клиент для запросов в Staff через StaffReader микросервис."""

    def __init__(self, url):
        self.url = url

    def request(self, url, query_params=None):
        """Отправка запроса в StaffReader."""
        try:
            resp = requests.get(
                url=url,
                params=query_params,
                verify=False,
                timeout=(settings.STAFF_READER_CONNECTION_TIMEOUT, settings.STAFF_READER_READ_TIMEOUT),
            )
        except ConnectionError as e:
            raise StaffReaderError('Connection Error: {}'.format(e))

        if resp.status_code != 200:
            raise StaffReaderError('Unexpected status {}'.format(resp.status_code))

        if 'application/json' not in resp.headers['content-type']:
            raise StaffReaderError('Response is not json')

        data = resp.json()
        if 'error_message' in data:
            raise StaffReaderError(data['error_message'])

        return data

    def get_suggestuser(self, username):
        """Запрос объекта пользователя."""
        CACHE_KEY = settings.STAFF_READER_CACHE_KEY_TMPL.format(username)

        data = main_cache.get(CACHE_KEY)

        if data is None:
            data = self.request(
                '{url}/suggestuser?login={username}'.format(
                    url=self.url,
                    username=username,
                ),
            )

            main_cache.set(
                CACHE_KEY,
                data,
                timeout=settings.STAFF_READER_CACHE_TTL,
            )

        return data

    def get_user_ui_language(self, username):
        """ Запрос языка интерфейса для пользователя ( должно влиять на язык интерфейса мебиуса ) """
        return self._get_user_languages_from_staff(
            username=username
        ).get(
            'ui',
            settings.STAFF_READER_DEFAULT_USER_LANGUAGE
        )

    def get_user_native_language(self, username):
        """ Запрос родного языка пользователя ( для использования в правилах назначения курсов) """
        return self._get_user_languages_from_staff(
            username=username
        ).get(
            'native',
            settings.STAFF_READER_DEFAULT_USER_LANGUAGE
        )

    def get_user_is_dismissed(self, username):
        return self.get_suggestuser(username=username).get(
            'official', {}
        ).get('is_dismissed', False)

    def _get_user_languages_from_staff(self, username):
        """ Запрос словаря языковых настроек пользователя из стафф-ридера или редис-кеша """
        CACHE_KEY = settings.USER_LANGUAGE_CACHE_KEY_TMPL.format(username)

        cached_language = None
        try:
            cached_language = main_cache.get(
                CACHE_KEY
            )
        except Exception as e:
            logger.error("Cannot get language from redis for user: {}. Exc: {}".format(
                username,
                str(e)
            ))

        if cached_language:
            return cached_language

        staff_languages = {
            'ui': settings.STAFF_READER_DEFAULT_USER_LANGUAGE,
            'native': settings.STAFF_READER_DEFAULT_USER_LANGUAGE,
        }

        try:
            data = self.request(
                '{url}/suggestuser?login={username}'.format(
                    url=self.url,
                    username=username,
                ),
            )
            staff_languages = data.get('language', {})

        except Exception as e:
            logger.warning("Cannot get languages from staff for user {}. Exc: {}".format(
                username,
                str(e)
            ))

        try:
            logger.debug("Storing user {} languages {} in cache.".format(username, staff_languages))
            main_cache.set(
                CACHE_KEY,
                staff_languages
            )
        except Exception as e:
            logger.warning("Cannot set languages cache for user: {}. Exc: {}".format(
                username,
                str(e)
            ))

        return staff_languages

    def get_suggestuser_many(self, usernames=None):
        """Множественный запрос объекта пользователя."""
        if usernames is None:
            return []
        results = []
        for chunk in chunks(usernames, 200):
            data = self.request(
                '{url}/suggestuser?login={username}&many=true'.format(
                    url=self.url,
                    username=','.join(chunk),
                ),
            )
            results += data['result']
        return results

    def get_user_groups(self, username):
        """Запрос групп пользователья."""
        response = self.request(
            '{url}?login={username}'.format(
                url=urljoin(self.url, 'usergroups'),
                username=username,
            )
        )

        if len(response) == 1 and 'groups' in response[0]:
            return response[0]['groups']

        return []

    def get_user_location(self, username):
        """Запрос местоположения пользователя."""
        response = self.request(
            '{url}?login={username}'.format(
                url=urljoin(self.url, 'userlocation'),
                username=username,
            )
        )

        if len(response) == 1 and 'location' in response[0]:
            return response[0]['location']

        return {}

    def get_user_groups_hierarchy(self, username):
        """
        Возвращает упорядоченный по уровню иерерхии массив групп пользователя, начинающийся с вернхней группы
        и заканчивающийся текущей, в которой напрямую состоит пользователь.
        """
        user_groups = self.get_user_groups(username)
        if not user_groups:
            return []

        # в общем случае считаем, что пользователь может состоять параллельно в более чем одной иерархии групп стаффа
        user_departments = [x for x in user_groups if x['group']['type'] == 'department']

        result = []

        for department in user_departments:
            current_group = department['group']
            ancestors = current_group.get('ancestors', None)

            if not ancestors:
                result.append([current_group['name']])
                continue

            hierarchy = [
                dep['name'] for dep in sorted(
                    ancestors,
                    key=lambda x: x['level'],
                    reverse=False
                )
            ] + [current_group['name']]
            result.append(hierarchy)

        return result

    def get_hrusers(self, username, limit=0, offset=0, with_nested: Optional[bool] = None):
        """Запрос всех подчиненных HR-партнера."""
        return self.get_subusers(
            url=urljoin(self.url, 'hrusers'),
            username=username,
            with_nested=with_nested,
            limit=limit,
            offset=offset,
        )

    def get_chiefusers(self, username, limit=0, offset=0, with_nested: Optional[bool] = None):
        """Запрос всех подчиненных руководителя."""
        return self.get_subusers(
            url=urljoin(self.url, 'chiefusers'),
            username=username,
            with_nested=with_nested,
            limit=limit,
            offset=offset,
        )

    def get_subusers(self, url, username, limit=0, offset=0, with_nested: Optional[bool] = None):
        """Запрос всех подопечнх пользователя."""
        query_params = {'login': username}
        if with_nested:
            query_params['with_nested'] = 'true'
        users = self.request(url=url, query_params=query_params)

        usernames = sorted(set([user['login'] for user in users]))
        if offset:
            usernames = usernames[offset:]
        if limit:
            usernames = usernames[:limit]

        #  For testing and development only
        if settings.ENVTYPE in [['development'], ['testing'], ['localhost']]:
            usernames = self.get_faked_subusers(usernames=usernames, username=username)

        subusers = self.get_suggestuser_many(usernames=usernames)
        return sorted(subusers, key=lambda subuser: subuser['login'])

    def get_user_roles(self, username):
        """Запрос staff-ролей пользователя"""
        try:
            roles = self.request('{url}/userroles?login={username}'.format(url=self.url, username=username))
        except StaffReaderError as e:
            logger.error(repr(e))
            return {'chief': False, 'hr_partner': False}

        roles['hr_partner'] = self.has_department_role(username=username, role='hr_partner')
        #  For testing and development only
        if settings.ENVTYPE in [['development'], ['testing'], ['localhost']]:
            roles = self.get_faked_user_roles(roles=roles, username=username)
        return roles

    def has_department_role(self, username, role):
        try:
            roles = self.request(
                u'{url}/hasdepartmentrole?login={username}&role={role}'.format(
                    url=self.url,
                    username=username,
                    role=role,
                )
            )
        except StaffReaderError as e:
            logger.error(repr(e))
            return False

        return roles.get('has_department_role', False)

    def get_faked_subusers(self, usernames, username):
        faked_user_group = Group.objects.filter(name=settings.FAKED_STAFF_USERS_GROUP).first()
        if not faked_user_group:
            return {'chief': False, 'hr_partner': False}
        faked_usernames = faked_user_group.user_set.all().values_list('username', flat=True)
        if username in faked_usernames:
            return list(faked_usernames)

        return usernames

    def get_faked_user_roles(self, roles, username):
        faked_user_group = Group.objects.filter(name=settings.FAKED_STAFF_USERS_GROUP).first()
        if not faked_user_group:
            return roles
        if username in faked_user_group.user_set.all().values_list('username', flat=True):
            return {'chief': True, 'hr_partner': True}
        return roles

    def get_all_groups(self):
        """Список все staff-групп."""
        limit = 100
        page = 1
        while True:
            response = self.request('{url}/groups?limit={limit}&page={page}'.format(
                url=self.url,
                limit=limit,
                page=page,
            ))
            page += 1
            for group in response['result']:
                yield group

            if response['page'] == response['pages']:
                return

    def get_all_offices(self):
        """Список всех staff-офисов с городами."""
        limit = 100
        page = 1
        while True:
            response = self.request('{url}/offices?limit={limit}&page={page}'.format(
                url=self.url,
                limit=limit,
                page=page,
            ))
            page += 1
            for group in response['result']:
                yield group

            if response['page'] == response['pages']:
                return

    def _get_auto_group_start_date(self, username, groupurl):
        """Запрос информации о дате присоединения к группе."""

        CACHE_KEY = settings.USER_WIKI_GROUPS_STAFF_READER_CACHE_KEY_TMPL.format(username)

        wiki_groups = main_cache.get(CACHE_KEY)
        if wiki_groups is None:
            wiki_groups = self.request(
                '{url}/autogroups?login={username}'.format(
                    url=self.url,
                    username=username,
                )
            )["result"]

            main_cache.set(
                CACHE_KEY,
                wiki_groups,
                timeout=settings.USER_WIKI_GROUPS_STAFF_READER_CACHE_TTL,
            )

        response = None
        for group in wiki_groups:
            if group['group'].get('url', '') == groupurl:
                response = re.sub(  # remove useless trailing pseudo timezone info
                    r"(\.\d{6})*\+\d+:\d+$",
                    "",
                    group['joined_at']
                )
                break  # no need to analyze rest of the groups
        return response

    def get_chief_start_date(self, username):
        """Запрос информации о дате присоединения к группе руководителей."""
        return self._get_auto_group_start_date(username, 'autoheads')

    def get_hr_partner_start_date(self, username):
        """Запрос информации о дате присоединения к группе HR-партнеров."""
        return self._get_auto_group_start_date(username, 'hr_partners')


class MockStaffReader(StaffReader):
    def get_user_native_language(self, username=''):
        return settings.STAFF_READER_DEFAULT_USER_LANGUAGE

    def get_user_ui_language(self, username=''):
        return settings.STAFF_READER_DEFAULT_USER_LANGUAGE

    def get_user_roles(self, username):
        return {'chief': False, 'hr_partner': False}

    def get_user_groups(self, username):
        return []

    def get_user_location(self, username):
        return {}

    def get_user_groups_hierarchy(self, username):
        return []

    def has_department_role(self, username, role):
        return True

    def get_suggestuser(self, username=''):
        return {
            'official': {
                'is_dismissed': False
            }
        }


def create_staff_reader_client():
    """Создание клиента для запросов в StaffReader"""
    staff_reader_class = getattr(
        sys.modules[__name__],
        settings.STAFF_READER_CLASS,
        StaffReader
    )
    client = staff_reader_class(settings.STAFF_READER_URL)

    return client


staff_reader = create_staff_reader_client()  # Глобальный инстанс StaffReader
