from collections import OrderedDict

from django.conf import settings
from django.template.base import Node
from django.template.defaultfilters import register


class AppOrderNode(Node):
    """
    Переупорядочивает список приложений и их моделей в админке. Основан на
    https://djangosnippets.org/snippets/2613/
    """

    def render(self, context):
        """
        Переупорядочивает приложения и их модели согласно `ADMIN_APP_ORDER`
        в настройках. Для остальных приложений пользуется порядком по умолчанию
        """
        if 'app_list' not in context:
            return ''
        reordered_apps = []
        app_map = OrderedDict(
            (app_def['app_label'], app_def)
            for app_def in context['app_list']
        )

        # Наполняем список приложений по порядку из `ADMIN_APP_ORDER`
        for app in settings.ADMIN_APP_ORDER:
            app_label, app_model_order = app[0], app[1]
            app_def = app_map.get(app_label)
            if not app_def:
                continue

            reordered_models = []
            model_map = OrderedDict(
                (model_def['object_name'], model_def)
                for model_def in app_def['models']
            )
            # Переупорядочиваем модели, аналогично - сначала по настройкам
            for model_name in app_model_order:
                model_def = model_map.get(model_name)
                if not model_def:
                    continue
                reordered_models.append(model_def)
                model_map.pop(model_name)

            reordered_models.extend(list(model_map.values()))
            app_def['models'] = reordered_models

            reordered_apps.append(app_def)
            app_map.pop(app_label)

        # Дополняем оставшимися приложениями и добавляем в контекст рендеринга
        reordered_apps.extend(list(app_map.values()))
        context['app_list'] = reordered_apps
        return ''


@register.tag
def app_order(parser, token):
    """
    Тег для переупорядочивания приложений в админке
    """
    return AppOrderNode()
