from builtins import object, zip

from past.builtins import basestring

from django.template.defaulttags import TemplateSyntaxError, register
from django.utils.translation import ugettext_lazy as _


class Genders(object):
    GENDER_MALE = 1
    GENDER_FEMALE = 2
    GENDER_UNSPECIFIED = 3

    GENDERS_DICT = {
        _('Мужской'): GENDER_MALE,
        _('Женский'): GENDER_FEMALE,
        _('Не указано'): GENDER_UNSPECIFIED,
    }


INDEX_BY_GENDER = dict(
    list(zip(
        list(Genders.GENDERS_DICT.values()),
        list(Genders.GENDERS_DICT.values()),
    ))
)
DEFAULT_INDEX = Genders.GENDER_UNSPECIFIED


@register.filter
def get_item(dictionary, key):
    """
    Returns dictionary[key] if `key` is in `dictionary`, else None
    """
    return dictionary.get(key)


@register.filter
def get_by_index(list_, index):
    """Return item in list by index"""
    return list_[index]


def get_pluralize_form(n):
    """
    Рассчитывает номер "русской" формы множественного числа.

    Правила (на примере слова "минута"):
    0 => минут
    1 => минута (форма 0)
    2..4 => минуты (форма 1)
    5..19 => минут (форма 2)
    20..99 => определяется последней цифрой
    100..+inf => определяется последними двумя цифрами

    В случае работы с десятичными дробями всегда используем форму 1:
    "пять целых, восемь сотых минуты"

    (copy-paste from cloner)
    """
    if isinstance(n, basestring):
        n = float(n)

    if int(n) != n:
        return 1

    n = abs(int(n)) % 100

    if n == 0 or 5 <= n < 20:
        return 2
    elif 1 <= n < 2:
        return 0
    elif 2 <= n < 5:
        return 1
    elif 20 <= n < 100:
        return get_pluralize_form(n % 10)


@register.filter
def rupluralize(value, arg):
    forms = arg.split(',')
    try:
        return forms[get_pluralize_form(value)]
    except Exception:
        raise TemplateSyntaxError


rupluralize.is_safe = False


@register.filter
def genderize(gender, arg):
    """
    Возвращает форму слова (из указанных через запятую в `args`),
    соответствующую полу `gender`.
    """
    forms = arg.split(',')
    try:
        return forms[INDEX_BY_GENDER.get(gender, DEFAULT_INDEX)]
    except Exception:
        raise TemplateSyntaxError


genderize.is_safe = False
