import datetime
import hashlib
import inspect
import json
import logging
import math
import os
import random
import re
import subprocess
import time
from builtins import object, range
from collections import Counter, OrderedDict
from functools import wraps
from itertools import groupby
from string import hexdigits
from subprocess import PIPE

import jsonschema
import pytz
from past.builtins import basestring
from unidecode import unidecode

from django.core.exceptions import ValidationError
from django.utils.functional import cached_property

ZERO_TIME = datetime.datetime.utcfromtimestamp(0).replace(tzinfo=pytz.utc)


ALPHABET = '0123456789ABCDEFGHKLMNPRSTUVWXYZ'
# символы, которые не совпадают с кириллическими буквами
# и не смущают пользователя
SAFE_ALPHABET = '23456789DFGLNRSUVWZ'

# Для фильтрации и замены "небезопасных" символов в именах файлов
UNSAFE_CHARACTERS = {
    ' ': '_',
    '\\': '',
    '\'': '',
    '/': '',
    ':': '',
    '?': '',
    '<': '',
    '>': '',
    '|': '',
    '!': '',
    '#': '',
    '$': '',
    '+': '',
    '%': '',
    '`': '',
    '&': '',
    '*': '',
    '{': '',
    '}': '',
    '"': '',
    '=': '',
    '@': '',
}

# Собственные транслитерации, обходящие таковые в таблицах `unidecode`
PRE_TRANSLITERATE = {
    u'й': 'y',
    u'ё': 'e',
    u'ъ': '',
    u'ь': '',
    u'ю': 'yu',
    u'я': 'ya',
    u'Й': 'Y',
    u'Ё': 'E',
    u'Ъ': '',
    u'Ь': '',
    u'Ю': 'Yu',
    u'Я': 'Ya',
}

COLOR_LENGTH = 7
COLOR_CREATION_ATTEMPTS = 2
DEFAULT_COLOR = '#50c878'

ANDROID_CLIENT_NAME = 'Android'

MAX_DEFAULT_FILE_NAME_LENGTH = 100

logger = logging.getLogger(__name__)


def generate_code(length, alphabet=None):
    """
    Сгенерировать буквенно-числовой код
    :param length: длина кода
    :return: код
    """
    return ''.join([random.choice(alphabet or ALPHABET)
                    for __ in range(length)])


def generate_safe_code(length):
    """
    Генерирует код из безопасного алфавита (нет похожих кириллических букв)
    """
    return generate_code(length, alphabet=SAFE_ALPHABET)


class EmptyTrueTuple(tuple):
    """
    Кортеж, который всегда логически истинное значение
    """

    def __bool__(self):
        return True


class OrderedCounter(Counter, OrderedDict):
    pass


def generate_graph_from_counters(answer, all_answers):
    """
    первый параметр - мультимножество
    второй параметр - мультимножество множеств

    Возвращает граф на основе ответов и возможных ответов. Если размерности не
    совпадают - возвращает False
    """

    # Используя `OrderCounter` мы сможем добиться того, чтобы результат
    # `.keys()` всегда был одинаковый (ключи шли в одном порядке)
    answer = OrderedCounter(answer)
    all_answers = OrderedCounter(all_answers)

    if sum(answer.values()) != sum(all_answers.values()):
        return False

    # Представление графа списком смежности. Нумерация вершин в двух долях
    # независимая. Индексом являются вершины введенного ответа (answer)
    graph = []
    for vertex_obj in list(answer.keys()):
        current_list_of_edges = []

        # Указатель на первую вершину следующего множества правильных ответов
        answer_num = 0
        for answer_option in list(all_answers.keys()):
            if vertex_obj in answer_option:
                # Добавляем грани до каждого вхождения найденной вершины.
                current_list_of_edges.extend(
                    list(range(answer_num, answer_num + all_answers[answer_option])))
            # смещаем указатель на первую вершину следующего множества
            # правильных ответов (увеличиваем на количество вхождений вершины).
            answer_num += all_answers[answer_option]

        for _ in range(answer[vertex_obj]):
            # добавляем найденные вершины столько раз,
            # сколько вершина встречается в ответе.
            graph.append(current_list_of_edges)
    return graph


def generate_random_color():
    """
    Случайный шестнадцатеричный RGB-цвет в виде строки
    """
    return '#' + ''.join(random.sample(hexdigits, COLOR_LENGTH - 1)).lower()


def generate_unique_color(existed_colors, additional_log_string='',
                          default_color=DEFAULT_COLOR,
                          count_attempts=COLOR_CREATION_ATTEMPTS):
    """
    Генерирует цвет, который не встречается в `existed_colors`
    При неудаче, возвращает дефолтный цвет и логирует неудачу с ипользованием
    `additional_log_string`
    :param count_attempts: количество попыток создания
    :param default_color: дефолтный цвет
    """
    for i in range(count_attempts):
        generated = generate_random_color()
        if generated not in existed_colors:
            return generated
    logger.warning('Had to assign default color %s', additional_log_string)
    return default_color


def generate_color_from_set(colors_limit, existed_colors=None):
    """
    Генерирует цвет из допустимого набора `colors_limit`
    Если все цвета уже задействованы, переходит на второй круг
    :param colors_limit: допустимый набор цветов, цвета с '#'
    :param existed_colors: уже использованные цвета
    """
    if not existed_colors:
        # если еще ни один цвет не использовали, берем любой
        return random.choice(list(colors_limit))
    most_common = Counter(existed_colors).most_common()
    usable_colors = set(colors_limit)
    max_count = most_common[0][1]
    for value, count in most_common:
        # если уже дошли до цветов, которые были использованы меньшее число
        # раз, значит, остальные цвета из предыдущих кругов
        if count < max_count:
            break
        else:
            usable_colors.discard(value)
    if len(usable_colors) == 0:
        # если все цвета использованы, переходим на следующий круг
        usable_colors = colors_limit
    return random.choice(list(usable_colors))


def group_level_sorted_key(group_level):
    """
    Специфический ключ для сортировки объектов `GroupLevel`: сначала
    по базовому уровню, затем по названию в алфавитном порядке с учетом чисел

    :param group_level: объект уровня группы
    :type group_level: GroupLevel
    """
    numbers_and_words = [int(''.join(group)) if isdigit else ''.join(group)
                         for isdigit, group
                         in groupby(group_level.name, str.isdigit)]
    return group_level.baselevel, numbers_and_words


def transliterate(to_transliterate):
    """
    Транслитерирует unicode-строку в ascii-символы. Пользуется unidecode,
    при этом часть символов изменяется заранее согласно `PRE_TRANSLITERATE`
    """
    result = ''.join(PRE_TRANSLITERATE.get(ch, ch) for ch in to_transliterate)
    return unidecode(result)


# TODO: использовать везде, где делаются таймстампы
def make_timestamp(datetime_obj):
    """
    Делает таймстамп из объекта datetime
    """
    return int(time.mktime(datetime_obj.timetuple()))


def safe_filename(filename, upload_dir=''):
    """
    Удаляет из имени файла "опасные" символы, заменяет пробелы на
    подчеркивания, транслитерирует его, добавляет в конец таймстамп
    для уникальности, укладывает его в MAX_DEFAULT_FILE_NAME_LENGTH символов
    """
    base_filename, extension = os.path.splitext(filename)
    base_filename = (
        ''.join(UNSAFE_CHARACTERS.get(ch, ch)
                for ch in unidecode(base_filename))
    )
    suffix = '_{0}{1}'.format(
        make_timestamp(datetime.datetime.now()), extension)
    slash_space = 1 if upload_dir and upload_dir[-1] != '/' else 0
    filename_crop = (MAX_DEFAULT_FILE_NAME_LENGTH - len(upload_dir) -
                     len(suffix) - slash_space)

    # Экстремальная ситуация, например гигантское расширение
    if filename_crop <= 0:
        raise ValueError('Either upload prefix or file extension are too big')

    return os.path.join(upload_dir, base_filename[:filename_crop] + suffix)


def dt_from_microseconds(microseconds):
    """
    Вернуть `datetime` из числа микросекунд
    """
    return (datetime.datetime.fromtimestamp(int(microseconds) / 1000000.).replace(tzinfo=pytz.utc))


def dt_to_microseconds(dt):
    """
    Вернуть число микросекунд в объекте `datetime`
    """
    return int((dt - ZERO_TIME).total_seconds() * 1000000)


def normalize_name(name):
    """
    Обрезает в строке начальные и конечные пробелы, заменяет множественные
    пробелы в середине одним, делает заглавной первую букву каждого слова,
    остальные делает строчными. Основное предназначение - имена, фамилии
    """
    return u' '.join(name.split()).title()


def get_percentage(dividend, divisor):
    """
    Получение целого процента из отношения двух целых чисел. При нулевом
    делителе возвращает 0
    """
    return (int(round(float(dividend) / divisor * 100))
            if dividend and divisor else 0)


def client_is_android(serializer_context):
    """
    По контексту сериализатора и запросу в нем определяет, является ли клиент
    android-приложением. Полагается на работу
    `kelvin.common.middleware.ClientNameMiddleware`
    """
    return (
        bool(serializer_context) and
        'request' in serializer_context and
        serializer_context['request'].META.get('client_application') == ANDROID_CLIENT_NAME
    )


def takewhile_with_first_fail(predicate, iterable):
    """
    Аналог `itertools.takewhile`, который вернет также первый элемент, где
    предикат будет ложным

    takewhile_with_first_fail(lambda x: x < 5, [1, 4, 6, 4, 1]) --> 1 4 6
    """
    for x in iterable:
        yield x
        if not predicate(x):
            # raise StopIteration
            return


def execute_cmd(cmd, stdin_text=None):
    """
    Executes `cmd` and returns it's output
    :param cmd: command as single string or as list of arguments
    :type cmd: str | list of str
    """
    process = subprocess.Popen(
        cmd,
        shell=isinstance(cmd, str),  # shell mode should be on if `cmd` is str
        stdin=PIPE,
        stdout=PIPE,
        stderr=PIPE,
        close_fds=True,
    )
    str_stdout, str_stderr = process.communicate(stdin_text)

    logger.debug('%s stdout: %s', cmd, str_stdout)
    logger.debug('%s stderr: %s', cmd, str_stderr)
    return str_stdout


def tipograph_string(string, inputs=None):
    """
    Изменяет строку по следующим правилам:

    * Изменить знак дефиса - на знак минус по следующим правилам: Соседи дефиса
    пробелы (1 или 2), и хотя бы один из соседей двух пробелов число или инпут
    типа число.
    ИЛИ Один сосед дефиса число или инпут типа число.
    * Кавычки "" заменить на нормальные елочки «»
    * Знак дефис заменяем на знак "длинное тире" (mdash) по следующим правилам:
    Соседи дефиса пробелы (1 или 2), и хотя бы один из соседей двух пробелов
    буква или инпут типа текст.
    """
    # Заменяем двойные кавычки на елочки
    text_string_array = string.split('"')
    odd_count_quotes = len(text_string_array) % 2 == 0
    new_text_string_array = []
    last_element_index = len(text_string_array) - 1
    for i, element in enumerate(text_string_array):
        new_text_string_array.append(element)
        if i != last_element_index:
            new_text_string_array.append(u'»' if i % 2 else u'«')
    string = ''.join(new_text_string_array)

    # Заменяем знак дефис (с клавиатуры) на знак минус (&#045;)
    int_inputs = [
        id_ for id_, value in list(inputs.items()) if
        value['type'] == 'field' and
        value['options']['type_content'] == 'number'
    ] if inputs else []
    int_input_str = ('(' + '|'.join(int_inputs) + ')') if int_inputs else ''
    string = re.sub(
        (
            '(((({{input:{int_input_str}}}|\d+)( {{1,2}}|\n)-( {{1,2}}|\n))'
            '|'
            '(( {{1,2}}|\n)-( {{1,2}}|\n)({{input:{int_input_str}}}|\d+)))'
            '|'
            '((({{input:{int_input_str}}}|\d+)-)'
            '|'
            '(-({{input:{int_input_str}}}|\d+))))'
        ).format(int_input_str=int_input_str),
        lambda find_obj: find_obj.group().replace('-', u'−'),
        string
    )

    # Заменяем знак дефис (с клавиатуры) на знак минус (&#045;) в таблице
    def replace_in_table(string):
        return re.sub(
            '\| *- *\|',
            lambda find_obj: find_obj.group().replace('-', u'−'),
            string
        )

    string = replace_in_table(replace_in_table(string))

    # Заменяем дефис (с клавиатуры) на знак длинного тире (mdash)
    text_inputs = [
        id_ for id_, value in list(inputs.items()) if
        value['type'] == 'field' and
        value['options']['type_content'] in ['text', 'spaceless', 'strict']
    ] if inputs else []
    text_input_str = ('(' + '|'.join(text_inputs) + ')') if text_inputs else ''
    string = re.sub(
        (
            u'(({{input:{text_input_str}}}|'
            u'([а-я]|ё))( {{1,2}}|\n)-( {{1,2}}|\n))'
            '|'
            u'(( {{1,2}}|\n)-( {{1,2}}|\n)'
            u'({{input:{text_input_str}}}|([а-я]|ё)))'
        ).format(text_input_str=text_input_str),
        lambda find_obj: find_obj.group().replace('-', u'—'),
        string
    )

    return string, odd_count_quotes


def create_jsonschema_validator(schema):
    """
    Возвращает функцию-валидатор для переданной JSON-схемы.
    Валидатор можно использовать в JSONField.
    """
    def validator(data):
        try:
            jsonschema.validate(data, schema)
        except jsonschema.ValidationError as e:
            raise ValidationError(e.message)

    return validator


class LoggableMixin(object):
    """
    Добавляет логгер в виде свойства класса
    """
    __loggers = {}

    @cached_property
    def log(self):
        name = u'{0}.{1}'.format(self.__module__, self.__class__.__name__)
        return logging.getLogger(name)  # type: logging.Logger


def log_method(method):
    """
    Декоратор логирует вызов функции или метода с параметрами
    """

    @wraps(method)
    def wrapper(*args, **kwargs):
        try:
            name = '{0}.{1}.{2}'.format(
                method.__module__, args[0].__class__.__name__, method.__name__
            )

        except IndexError:
            name = '{0}.{1}'.format(method.__module__, method.__name__)

        start_time = time.time()
        result = method(*args, **kwargs)
        total_time = (time.time() - start_time)

        log = logging.getLogger(name)
        arguments = inspect.getfullargspec(method).args
        log.debug('Call: %s(%s) in %s', method.__name__, arguments, total_time)

        return result

    return wrapper


def positive_int_or_400(param, message=None):
    """
    Raises validation error (resulting in 400 Bad Request) if parameter is not
    positive int
    """
    if isinstance(param, basestring):
        if param.isdigit():
            param = int(param)
    else:
        param = param

    if isinstance(param, int) and param > 0:
        return param

    raise ValidationError(message or 'Expected positive int')


def old_round(x, d=0):
    p = 10 ** d
    if x > 0:
        return float(math.floor((x * p) + 0.5)) / p
    else:
        return float(math.ceil((x * p) - 0.5)) / p


def json_hash(json_data):
    dump = json.dumps(json_data, sort_keys=True, indent=2)
    return hashlib.md5(dump.encode('utf-8')).hexdigest()
