from django.contrib import admin
from django.utils.encoding import force_text
from django.utils.translation import ugettext_lazy as _


class ClosedCoursesFilter(admin.SimpleListFilter):
    """
    Фильтрация по закрытости курсов
    """
    title = u'Закрытость курсов'
    parameter_name = 'closed_course'
    LOOKUPS = (
        ('1', u'Открыт'),
        ('2', u'Закрыт'),
        ('3', u'Все'),
    )

    def queryset(self, request, queryset):
        """
        Фильтрация значений по выбранному фильтру
        """
        if self.value() == self.LOOKUPS[0][0] or self.value() is None:
            return queryset.filter(date_closed__isnull=True)
        elif self.value() == self.LOOKUPS[1][0]:
            return queryset.filter(date_closed__isnull=False)
        elif self.value() == self.LOOKUPS[2][0]:
            return queryset

    def lookups(self, request, model_admin):
        """
        Список возможных фильтров
        """
        return self.LOOKUPS

    def choices(self, cl):
        """Первое значение делаем выбранным по умолчанию"""
        lookup, title = self.lookup_choices[0]
        yield {
            'selected': (
                self.value() is None or self.value() == force_text(lookup)),
            'query_string': cl.get_query_string(
                {self.parameter_name: lookup}, []),
            'display': title,
        }
        for lookup, title in self.lookup_choices[1:]:
            yield {
                'selected': self.value() == force_text(lookup),
                'query_string': cl.get_query_string({
                    self.parameter_name: lookup,
                }, []),
                'display': title,
            }


class RawIdCourseFilter(admin.ListFilter):
    """
    Фильтр в виде формы для идентификатора курса
    """
    title = _(u'Идентификатор курса')
    parameter_name = 'course_id'
    template = 'admin/courses/raw_id_course_filter.html'

    def __init__(self, request, params, model, model_admin):
        super(RawIdCourseFilter, self).__init__(
            request, params, model, model_admin)
        self.value = None
        if self.parameter_name in params:
            self.value = params.pop(self.parameter_name)

    def expected_parameters(self):
        return [self.parameter_name]

    def has_output(self):
        return True

    def choices(self, cl):
        return [self.value] if self.value else []

    def queryset(self, request, queryset):
        try:
            value = int(self.value)
        except (TypeError, ValueError):
            return queryset
        return queryset.filter(course_id=value)
