from builtins import object

from django import forms
from django.contrib.auth import get_user_model
from django.utils import timezone

from kelvin.common.admin import InlineJSONFormField, OwnerOnlyTeacherModelFormMixin
from kelvin.common.form_mixins import InfoFormMixin
from kelvin.courses.models import Course, CourseLessonLink
from kelvin.courses.services import copy_course

User = get_user_model()


class CourseForm(InfoFormMixin,
                 OwnerOnlyTeacherModelFormMixin,
                 forms.ModelForm):
    class Meta(object):
        model = Course
        fields = (
            'name',
            'subject',
            'project',
            'mode',
            'owner',
            'source_courses',
            'author',
            'color',
            'cover',
            'copy_of',
            'group_levels',
            'free',
            'copyright',
            'description',
            'code',
            'date_closed',
            'allow_anonymous',
            'news',
            'info',
            'progress_indicator',
            'available_for_support',
        )

    additional_info_props = ['company', 'links']

    def info_company_schema(self):
        return {
            'type': 'object',
            'properties': {
                'logo': {
                    'type': 'string',
                },
                'link': {
                    'type': 'string',
                },
                'name': {
                    'type': 'string',
                },
                'text': {
                    'type': 'string',
                },
            },
            'additionalProperties': False,
            'required': ['logo', 'link', 'name'],
        }

    def info_company_help_text(self):
        return (
            u"""{
                "logo": "url_with_picture",
                "link": "url_link",
                "name": "Компания",
                "text": "О компании"
            }"""
        )

    def info_links_schema(self):
        return {
            'type': 'object',
            'additionalProperties': {'type': 'string'},
            'minProperties': 1,
        }

    def info_links_help_text(self):
        return (
            u"""{
                "link": "url_link"
            }"""
        )


class CopyCourseForm(OwnerOnlyTeacherModelFormMixin, forms.ModelForm):
    """
    Форма для задания владельца копии курса
    """
    class Meta(object):
        model = Course
        fields = (
            'owner',
        )

    def __init__(self, *args, **kwargs):
        super(CopyCourseForm, self).__init__(*args, **kwargs)
        self.fields['owner'].queryset = User.objects.filter(is_teacher=True)

    def save(self, commit=True):
        """
        Копируем курс
        """
        copy_course(self.instance, owner=self.cleaned_data['owner'])
        return self.instance


class CourseLessonLinkInlineForm(forms.ModelForm):
    url = InlineJSONFormField(required=False, label=u'Адреса для вебинара')

    class Meta:
        model = CourseLessonLink
        fields = '__all__'


class CopiedCourseAccessForm(forms.ModelForm):
    """
    Форма для предоставления доступа к занятию учителю в курсе-копии
    """
    id = forms.CharField(widget=forms.HiddenInput())

    class Meta(object):
        model = Course
        fields = (
            'id',
        )

    def save(self, commit=True):
        """Делаем доступным занятие учителю"""
        clesson = self.instance.courselessonlink_set.filter(
            accessible_to_teacher=None,
            copy_of__isnull=False,
        ).order_by('order', 'id').first()
        if clesson:
            clesson.accessible_to_teacher = timezone.now()
            clesson.save()
        return self.instance
